import * as userModel from "../models/userModel.js";


export const formatUrl = (fileName) => {
    const BASE_URL = process.env.BASE_URL;
    if (!fileName || fileName === 'null') return null;
    const isFullUrl = fileName.startsWith('http://') || fileName.startsWith('https://');
    if (isFullUrl) return fileName;
    const hasExtension = /\.[a-zA-Z0-9]+$/i.test(fileName);
    if (!hasExtension) {
        fileName += '.png';
    }
    return `${BASE_URL}/uploads/${fileName}`;
};

export const formatScore = (score) => {
    if (score === null || score === undefined) return null;
    const num = parseFloat(score);
    return Number.isInteger(num)
        ? num.toString()
        : num.toFixed(2).toString();
};

export const generateOtp = (length = 6) => {
    let otp = '';
    for (let i = 0; i < length; i++) {
        otp += Math.floor(Math.random() * 10);
    }
    return otp;
};

export const calculateEssentialScore = ({ essentials_id, value, value2, gender, age }) => {

    const parseValue = (str) => {
        if (!str) return 0;
        const match = str.match(/\d+/);
        return match ? parseInt(match[0]) : 0;
    };

    const val = parseValue(value);
    const val2 = (essentials_id === 5 || essentials_id === 6) ? parseValue(value2) : 0;

    let score = 0;

    switch (essentials_id) {

        case 1: // Sleep (15 points)
            if (val >= 7 && val <= 9) score = 15;
            else if (val < 7) score = Math.max(15 - (7 - val) * 3, 0);
            else score = Math.max(15 - (val - 9) * 3, 0);
            break;

        case 2: // Sun exposure (10 points)
            score = Math.max(10 - Math.ceil(Math.abs(30 - val) / 5) * 2, 0);
            break;

        case 3: // Physical Activity / Exercise (15 points)
            score = Math.max(15 - Math.ceil(Math.max(30 - val, 0) / 5) * 3, 0);
            break;

        case 4: // Social engagement (15 points)
            score = val === 0 ? 0 : 15;
            break;

        case 5: // Food - total meals
        case 6: // Clean meals
            const totalMeals = val;
            const cleanMeals = val2;
            if (totalMeals > 0) {
                score = (cleanMeals / totalMeals) * 15;  // percentage * 10
                if (totalMeals < 3) {
                    score = Math.max(score - (3 - totalMeals) * 5, 0);  // deduct 5 points per missing meal
                }
            } else {
                score = 0;
            }
            break;

        case 7: // Water (5 points)
            const minWater = gender === 1 ? 125 : 91; // Male = 1, Female = 0
            score = Math.max(5 - Math.floor(Math.max(minWater - val, 0) / 15), 0);
            break;

        case 8: // Alcohol (5 points)
            if (gender === 0) score = val <= 1 ? 5 : Math.max(5 - (val - 1) * 2.5, 0);
            else score = val <= 2 ? 5 : Math.max(5 - (val - 2) * 2.5, 0);
            break;

        case 9: // Substances (5 points)
            score = value.toLowerCase() === "yes" ? 0 : 5;
            break;

        case 10: // Screen time (10 points)
            const maxScreen = age === 0 ? 120 : 240;  // under 18 vs adult   0 = Under 18, 1 = 18+
            score = Math.max(10 - Math.floor(Math.max(val - maxScreen, 0) / 30) * 2.5, 0);
            break;

        case 11: // Pornography (5 points)
            score = value.toLowerCase() === "yes" ? 0 : 5;
            break;

        default:
            score = 0;
    }

    return score;
};

// export const fetchProgress = async (user_id) => {

//     const essentials = await userModel.essentialsListing();
//     const responses = await userModel.getUserEssentialsWithScores(user_id);

//     const now = new Date();
//     const weekMap = {};       // store scores per week
//     const essentialMap = {};  // store scores per essential

//     // Map responses
//     responses.forEach(r => {
//         const score = r.score || 0;

//         // --- Weekly aggregation ---
//         if (r.date) {
//             const date = new Date(r.date * 1000);
//             const diffWeeks = Math.floor((now - date) / (7 * 24 * 60 * 60 * 1000));
//             const week = 5 - diffWeeks; // last 5 weeks
//             if (week >= 1 && week <= 5) {
//                 if (!weekMap[week]) weekMap[week] = [];
//                 weekMap[week].push(score);
//             }
//         }

//         // --- Essential aggregation ---
//         if (!essentialMap[r.essentials_id]) {
//             essentialMap[r.essentials_id] = {
//                 name: r.name,
//                 image: r.image,
//                 description: r.description,
//                 total_score: r.total_score,
//                 scores: []
//             };
//         }
//         essentialMap[r.essentials_id].scores.push(score);
//     });

//     // --- Weekly averages (last 5 weeks) ---
//     // const weeklyScores = Array.from({ length: 5 }, (_, i) => {
//     //     const week = i + 1;
//     //     const scores = weekMap[week] || [];
//     //     const avg = scores.length
//     //         ? parseFloat((scores.reduce((a, b) => a + b, 0) / scores.length).toFixed(1))
//     //         : 0;
//     //     return { week: `Week ${week}`, average: avg };
//     // });
// const weeklyScores = Array.from({ length: 5 }, (_, i) => {
//     const week = i + 1;
//     const scores = weekMap[week] || [];
//     const avg = scores.length
//         ? parseFloat((scores.reduce((a, b) => a + b, 0) / scores.length).toFixed(1))
//         : 0;

//     return {
//         week: `Week ${week}`,
//         average: avg.toString()   // ✅ returned as STRING
//     };
// });

//     // --- Determine improvedAreas & focusAreas ---
//     const improvedAreas = [];
//     const focusAreas = [];

//     essentials.forEach(e => {
//         const userData = essentialMap[e.essentials_id];
//         const avg_score = userData && userData.scores.length
//             ? userData.scores.reduce((a, b) => a + b, 0) / userData.scores.length
//             : 0;

//         if (avg_score >= 0.7 * e.total_score) {
//             improvedAreas.push({
//                 essentials_id: e.essentials_id,
//                 name: e.essential_name,
//                 image: e.essential_image,
//                 suggestion: e.essential_description
//             });
//         } else {
//             focusAreas.push({
//                 essentials_id: e.essentials_id,
//                 name: e.essential_name,
//                 image: e.essential_image,
//                 suggestion: e.essential_description
//             });
//         }
//     });

//     return { weeklyScores, improvedAreas, focusAreas };
// };

export const fetchProgress = async (user_id, month_epoch) => {

    const essentials = await userModel.essentialsListing();
    const responses = await userModel.getUserEssentialsWithScores(user_id);

    // -----------------------------------------------------------
    // 1️⃣ FIXED MONTH FILTER (month_epoch ALWAYS SENT)
    // -----------------------------------------------------------
    const monthStart = new Date(month_epoch * 1000);  // month start
    const year = monthStart.getUTCFullYear();
    const month = monthStart.getUTCMonth();

    const nextMonthStart = new Date(Date.UTC(year, month + 1, 1));

    const filteredResponses = responses.filter(r => {
        if (!r.date) return false;

        const d = new Date(r.date * 1000);
        return d >= monthStart && d < nextMonthStart;
    });

    // -----------------------------------------------------------
    // 2️⃣ CALCULATE TOTAL WEEKS IN THAT MONTH
    // -----------------------------------------------------------
    const daysInMonth = new Date(Date.UTC(year, month + 1, 0)).getUTCDate();
    const totalWeeks = Math.ceil(daysInMonth / 7);

    const weekMap = {};
    const essentialMap = {};

    // -----------------------------------------------------------
    // 3️⃣ GROUP SCORES BY WEEK INSIDE THE MONTH
    // -----------------------------------------------------------
    filteredResponses.forEach(r => {
        const score = r.score || 0;

        const d = new Date(r.date * 1000);
        const day = d.getUTCDate();

        const weekNumber = Math.ceil(day / 7); // 1–4/5

        if (!weekMap[weekNumber]) weekMap[weekNumber] = [];
        weekMap[weekNumber].push(score);

        // Aggregate essential-wise
        if (!essentialMap[r.essentials_id]) {
            essentialMap[r.essentials_id] = {
                name: r.name,
                image: r.image,
                description: r.description,
                total_score: r.total_score,
                scores: []
            };
        }
        essentialMap[r.essentials_id].scores.push(score);
    });

    // -----------------------------------------------------------
    // 4️⃣ CALCULATE WEEKLY AVERAGES DYNAMICALLY
    // -----------------------------------------------------------
    const weeklyScores = Array.from({ length: totalWeeks }, (_, i) => {
        const w = i + 1;
        const scores = weekMap[w] || [];

        const avg = scores.length
            ? parseFloat((scores.reduce((a, b) => a + b, 0) / scores.length).toFixed(1))
            : 0;

        return {
            week: `Week ${w}`,
            average: avg.toString()
        };
    });

    // -----------------------------------------------------------
    // 5️⃣ IMPROVED AREAS / FOCUS AREAS (ONLY THIS MONTH)
    // -----------------------------------------------------------
    const improvedAreas = [];
    const focusAreas = [];

    essentials.forEach(e => {
        const data = essentialMap[e.essentials_id];

        const avg_score = data && data.scores.length
            ? data.scores.reduce((a, b) => a + b, 0) / data.scores.length
            : 0;

        if (avg_score >= 0.7 * e.total_score) {
            improvedAreas.push({
                essentials_id: e.essentials_id,
                name: e.essential_name,
                image: e.essential_image,
                suggestion: e.essential_description,
                total_score: e.total_score,
                score: avg_score
            });
        } else {
            focusAreas.push({
                essentials_id: e.essentials_id,
                name: e.essential_name,
                image: e.essential_image,
                suggestion: e.essential_description,
                total_score: e.total_score,
                score: avg_score
            });
        }
    });

    return {
        weeklyScores,
        improvedAreas,
        focusAreas
    };
};