import { OAuth2Client } from "google-auth-library";
import * as subscriptionModel from "../../models/subscriptionModel.js";

const BASE_URL = process.env.BASE_URL;
const CLIENT_ID = process.env.GOOGLE_CLIENT_ID;
const CLIENT_SECRET = process.env.GOOGLE_CLIENT_SECRET;
const REDIRECT_URI = `${BASE_URL}/api/v1/subscription/google-callback`;
const SCOPE = "https://www.googleapis.com/auth/androidpublisher";

const oauth2Client = new OAuth2Client({
    clientId: CLIENT_ID,
    clientSecret: CLIENT_SECRET,
    redirectUri: REDIRECT_URI,
});

const getAccessToken = async () => {
    try {
        const tokens = await subscriptionModel.getGoogleTokens();
        if (!tokens.length) throw new Error("No Google tokens found");

        const tokenData = tokens[0];
        const now = Math.floor(Date.now() / 1000);

        if (tokenData.access_token_expires_at > now) return tokenData.access_token;
        if (!tokenData.refresh_token) throw new Error("Refresh token missing");

        oauth2Client.setCredentials({ refresh_token: tokenData.refresh_token });
        const accessTokenResponse = await oauth2Client.getAccessToken();
        const accessToken = accessTokenResponse.token;

        if (!accessToken) throw new Error("Failed to refresh access token");

        const accessTokenExpiresAt = Math.floor(oauth2Client.credentials.expiry_date / 1000);
        const newRefreshToken = oauth2Client.credentials.refresh_token || tokenData.refresh_token;

        await subscriptionModel.storeGoogleTokens(
            accessToken,
            newRefreshToken,
            accessTokenExpiresAt,
            tokenData.refresh_token_expires_at
        );

        return accessToken;
    } catch (error) {
        console.error("Error in getAccessToken:", error);
        throw error;
    }
};

export { oauth2Client, SCOPE, getAccessToken };