import axios from "axios";
import apn from "apn";
import { readFileSync } from "fs";
import { join, dirname } from "path";
import { fileURLToPath } from "url";
import { JWT } from "google-auth-library";
import logger from "../../utils/logger.js";

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

// ---------- Config Paths ----------
const APN_KEY_PATH = join(__dirname, "../../config/certs/appleCredentials.p8");
const SERVICE_ACCOUNT_PATH = join(__dirname, "../../config/certs/androidCredentials.json");

// ---------- Load Service Account ----------
const serviceAccount = JSON.parse(readFileSync(SERVICE_ACCOUNT_PATH, "utf-8"));

// ---------- Validate Environment Variables ----------
["APN_KEY_ID", "APN_TEAM_ID", "APN_TOPIC"].forEach((key) => {
  if (!process.env[key]) throw new Error(`Missing environment variable: ${key}`);
});

// ---------- Default Notification ----------
const DEFAULT_NOTIFICATION = {
  title: "",
  message: "",
  pushType: "0",
  imageUrl: "",
  sound: "default",
};

// ---------- Helpers ----------
const delay = (ms) => new Promise((res) => setTimeout(res, ms));
const normalizeTokens = (tokens) => (Array.isArray(tokens) ? tokens.filter(Boolean) : tokens ? [tokens] : []);

// ---------- Google JWT Client ----------
const gcmClient = new JWT({
  email: serviceAccount.client_email,
  key: serviceAccount.private_key,
  scopes: ["https://www.googleapis.com/auth/firebase.messaging"],
});

let cachedToken = null;
let tokenExpiry = 0;
const getAccessToken = async () => {
  const now = Date.now();
  if (cachedToken && now < tokenExpiry - 60000) return cachedToken;
  const { access_token, expiry_date } = await gcmClient.authorize();
  cachedToken = access_token;
  tokenExpiry = expiry_date;
  return access_token;
};

// ---------- Axios Instance for FCM ----------
const fcmAxios = axios.create({
  baseURL: `https://fcm.googleapis.com/v1/projects/${serviceAccount.project_id}/`,
  headers: { "Content-Type": "application/json" },
  timeout: 15000,
});

// ---------- APN Provider Singleton ----------
const apnProvider = new apn.Provider({
  token: {
    key: APN_KEY_PATH,
    keyId: process.env.APN_KEY_ID,
    teamId: process.env.APN_TEAM_ID,
  },
  production: process.env.APN_PRODUCTION === "true",
});

// ---------- Enterprise Push Handler ----------
async function processInBatches(items, batchSize = 100, concurrency = 5, handler, delayBetweenBatches = 0) {
  let totalSuccess = 0;
  let failedItems = [];

  for (let i = 0; i < items.length; i += batchSize * concurrency) {
    const batchGroups = [];

    for (let j = 0; j < concurrency; j++) {
      const batch = items.slice(i + j * batchSize, i + (j + 1) * batchSize);
      if (batch.length) batchGroups.push(batch);
    }

    const groupResults = await Promise.all(batchGroups.map(handler));

    groupResults.forEach((res) => {
      totalSuccess += res.successCount;
      failedItems.push(...res.failedItems);
    });

    if (delayBetweenBatches) await delay(delayBetweenBatches);
  }

  return { total: items.length, success: totalSuccess, failure: failedItems.length, failedItems };
}

// ---------- Android Push (Enterprise) ----------
export const sendAndroidPush = async (data = {}, deviceTokens, batchSize = 100, concurrency = 5) => {
  const tokens = normalizeTokens(deviceTokens);
  if (!tokens.length) return { total: 0, success: 0, failure: 0, failedItems: [] };

  const payload = { ...DEFAULT_NOTIFICATION, ...data };
  const token = await getAccessToken();

  const handler = async (batch) => {
    const results = await Promise.allSettled(
      batch.map(async (deviceToken) => {
        const message = {
          message: {
            token: deviceToken,
            data: {
              title: payload.title,
              message: payload.message,
              pushType: String(payload.pushType),
              imageUrl: payload.imageUrl,
              sound: payload.sound,
            },
          },
        };
        for (let attempt = 1; attempt <= 3; attempt++) {
          try {
            await fcmAxios.post("messages:send", message, { headers: { Authorization: `Bearer ${token}` } });
            logger.info(`✅ Android push sent to ${deviceToken}`);
            return { success: true };
          } catch (err) {
            const status = err.response?.status;
            const retryable = !status || (status >= 500 && status < 600);
            logger.warn(`⚠️ Android push attempt ${attempt} failed for ${deviceToken}: ${err.message}`);
            if (attempt < 3 && retryable) await delay(500 * Math.pow(2, attempt - 1));
            else {
              logger.error(`❌ Android push failed for ${deviceToken}: ${JSON.stringify(err.response?.data || err.message)}`);
              return { success: false, token: deviceToken, error: err.response?.data || err.message };
            }
          }
        }
      })
    );

    const successCount = results.filter(r => r.status === "fulfilled" && r.value?.success).length;
    const failedItems = results
      .filter(r => r.status === "fulfilled" && !r.value.success)
      .map(r => ({ token: r.value.token, error: r.value.error }));

    logger.info(`Android batch completed: ${successCount} succeeded, ${failedItems.length} failed`);
    return { successCount, failedItems };
  };

  return processInBatches(tokens, batchSize, concurrency, handler);
};

// ---------- Apple Push (Enterprise) ----------
export const sendApplePush = async (data = {}, deviceTokens, batchSize = 100, concurrency = 5) => {
  const tokens = normalizeTokens(deviceTokens);
  if (!tokens.length) return { total: 0, success: 0, failure: 0, failedItems: [] };

  const payload = { ...DEFAULT_NOTIFICATION, ...data };

  const handler = async (batch) => {
    const notification = new apn.Notification({
      alert: { title: payload.title, body: payload.message },
      sound: payload.sound,
      payload,
      topic: process.env.APN_TOPIC,
    });

    const result = await apnProvider.send(notification, batch);

    const successCount = result.sent?.length || 0;
    const failedItems = (result.failed || []).map(f => ({ token: f.device, error: f.response?.reason || "Unknown" }));

    if (successCount > 0) logger.info(`✅ Apple push sent to ${successCount} devices`);
    if (failedItems.length > 0) logger.error(`❌ Apple push failed for ${failedItems.map(f => f.token).join(", ")}`);

    return { successCount, failedItems };
  };

  return processInBatches(tokens, batchSize, concurrency, handler, 50);
};

// ---------- Cleanup ----------
const cleanup = () => {
  logger.info("🧹 Shutting down APN provider...");
  apnProvider.shutdown();
};

["SIGINT", "SIGTERM", "exit"].forEach(event => process.on(event, cleanup));