import cron from "node-cron";
import * as userModel from "../../models/userModel.js";
import { sendAndroidPush, sendApplePush } from "../../services/notifications/pushNotificationService.js";


// export const essentialsReminderCron = () => {
//   cron.schedule("* * * * *", async () => {
//     console.log("🔁 [CRON] Running Essentials Reminder Job...");
//     try {
//       const now = new Date();
//       const startOfDay = Math.floor(now.setHours(0, 0, 0, 0) / 1000);
//       const endOfDay = Math.floor(now.setHours(23, 59, 59, 999) / 1000);

//       const users = await userModel.getUsersWithIncompleteEssentials(startOfDay, endOfDay);

//       const notificationPayload = {
//         title: "Complete Your Essentials",
//         message: "You have pending essential responses for today. Fill them to track your progress!",
//         pushType: "2"
//       };

//       const androidTokens = [];
//       const iosTokens = [];
//       const notificationRecords = users.map(user => {
//         if (user.device_type === "Android") androidTokens.push(user.device_token);
//         else if (user.device_type === "Apple") iosTokens.push(user.device_token);
//         return {
//           user_id: user.id,
//           title: notificationPayload.title,
//           message: notificationPayload.message,
//           pushType: notificationPayload.pushType
//         };
//       });

//       await Promise.allSettled([
//         androidTokens.length ? sendAndroidPush(notificationPayload, androidTokens) : Promise.resolve(),
//         iosTokens.length ? sendApplePush(notificationPayload, iosTokens) : Promise.resolve()
//       ]);

//       await userModel.insertNotifications(notificationRecords);

//       console.log(`✅ Essentials reminders sent to ${users.length} users.`);
//     } catch (error) {
//       console.error("❌ Error in Essentials Reminder Cron:", error);
//     }
//   });
// };

export const essentialsReminderCron = () => {
  cron.schedule("0 * * * *", async () => {
    console.log("🔁 [CRON] Running Essentials Reminder Job...");
    try {
      const now = new Date();
      const startOfDay = Math.floor(new Date(now.setHours(0, 0, 0, 0)) / 1000);
      const endOfDay = Math.floor(new Date(now.setHours(23, 59, 59, 999)) / 1000);
      const currentDayName = now.toLocaleDateString("en-US", { weekday: "long" });

      const users = await userModel.getUsersWithIncompleteEssentials(startOfDay, endOfDay);

      if (!users.length) {
        console.log("ℹ️ No users found for essentials reminder.");
        return;
      }

      for (const user of users) {
        const userEssentials = await userModel.getEssentials(
          user.id,
          startOfDay,
          endOfDay,
          currentDayName
        );

        const incompleteEssentials = userEssentials
          .filter(e => e.value === null)
          .map(e => e.essential_name);

        if (!incompleteEssentials.length) continue;

        const notificationPayload = {
          title: "Complete Your Essentials",
          message: `You still need to complete: ${incompleteEssentials.join(", ")}`,
          pushType: "2",
        };

        if (user.device_type === "Android")
          await sendAndroidPush(notificationPayload, [user.device_token]);
        else if (user.device_type === "Apple")
          await sendApplePush(notificationPayload, [user.device_token]);

        await userModel.insertNotifications([
          {
            user_id: user.id,
            ...notificationPayload,
          },
        ]);
      }

      console.log(`✅ Essentials reminders sent to ${users.length} users.`);
    } catch (error) {
      console.error("❌ Error in Essentials Reminder Cron:", error);
    }
  });
};