import cron from "node-cron";
import axios from "axios";
import * as subscriptionModel from "../../models/subscriptionModel.js";
import { getAccessToken } from "../../services/subscription/subscriptionService.js";


export const checkAndroidSubscriptionsCron = () => {
    cron.schedule("* * * * *", async () => {
        console.log("🔄 Running Android subscription check cron...");
        try {
            const activeSubscriptions = await subscriptionModel.getActiveAndroidSubscriptions();
            if (!activeSubscriptions.length) return console.log("No active Android subscriptions");

            const accessToken = await getAccessToken();
            const now = Math.floor(Date.now() / 1000);

            for (const sub of activeSubscriptions) {
                const { user_id, id, package_name, purchase_token, expires_at, plan_id, plan_price, device_type } = sub;
                try {
                    const apiUrl = `https://androidpublisher.googleapis.com/androidpublisher/v3/applications/${package_name}/purchases/subscriptionsv2/tokens/${purchase_token}`;
                    const { data } = await axios.get(apiUrl, {
                        headers: { Authorization: `Bearer ${accessToken}` },
                    });

                    const lineItem = data.lineItems?.[0];
                    if (!lineItem) continue;

                    const expiryTime = lineItem.expiryTime ? Math.floor(new Date(lineItem.expiryTime).getTime() / 1000) : null;
                    const autoRenewing = lineItem.autoRenewingPlan?.autoRenewEnabled;

                    // Subscription expired and not auto-renewing
                    if (expiryTime && now > expiryTime && !autoRenewing) {
                        await subscriptionModel.updateSubscriptionStatus(user_id, id, "expired", "expired");
                        console.log(`Subscription expired for user ${user_id}, subscription ${id}`);
                    }

                    // Subscription renewed
                    if (autoRenewing && expiryTime && expiryTime > expires_at) {
                        // Update previous record to expired
                        await subscriptionModel.updateSubscriptionStatus(user_id, id, "expired", "renewed");

                        // Insert new subscription row for renewed plan
                        await subscriptionModel.insertSubscription({
                            user_id,
                            plan_id,
                            subscription_id: sub.subscription_id,
                            expires_at: expiryTime,
                            plan_price,
                            product_id: sub.product_id,
                            device_type,
                            package_name,
                            order_id: sub.order_id,
                            purchase_token,
                            transaction_id: sub.transaction_id,
                            plan_duration: sub.plan_duration,
                            status: "active",
                            event_type: "renewed",
                        });
                        console.log(`Subscription renewed for user ${user_id}, new expiry ${expiryTime}`);
                    }
                } catch (error) {
                    console.log(`Error processing subscription for user ${user_id}: ${error.message}`);
                }
            }
            console.log("✅ Android subscription check cron completed");
        } catch (error) {
            console.log("Error in Android subscription cron: " + error.message);
        }
    });
};