import pool from "../config/database.js";
import { Tables } from "../utils/dbTables.js";
import { formatUrl, formatScore } from "../utils/commonFunctions.js";


export const essentialsListing = async () => {
    try {
        const [rows] = await pool.execute(
            `SELECT id as essentials_id, essential_name, essential_image, essential_description, total_score
             FROM ${Tables.ESSENTIALS} 
             WHERE deleted_at IS NULL 
             ORDER BY id ASC`
        );
        const data = rows.map(row => ({
            ...row,
            essential_image: row.essential_image ? formatUrl(row.essential_image) : null
        }));
        return data;
    } catch (error) {
        console.error("Error in essentialsListing model:", error);
        throw error;
    }
};

export const updateUserById = async (user_id, updates) => {
    try {
        const fields = Object.keys(updates).map(k => `${k} = ?`).join(", ");
        const values = [...Object.values(updates), user_id];
        await pool.execute(`UPDATE users SET ${fields}, updated_at = NOW() WHERE id = ?`, values);
        return;
    } catch (error) {
        console.error("Error in updateUserById:", error.message);
        throw new Error("Database query failed");
    }
};

export const deleteEssentialsByCount = async (user_id, on_boarding_count) => {
    try {
        await pool.execute(
            `DELETE FROM essentials_responses 
             WHERE user_id = ? 
             AND essentials_id = ?`,
            [user_id, on_boarding_count]
        );
    } catch (error) {
        console.error("Error deleting essentials responses:", error);
        throw error;
    }
};

export const getQuestionWithOptions = async (questionId, userId) => {
    try {
        const [questions] = await pool.execute(
            `SELECT id, question, warning_heading, warrning 
             FROM questions 
             WHERE id = ? AND deleted_at IS NULL`,
            [questionId]
        );

        if (questions.length === 0) return null;

        const [options] = await pool.execute(
            `SELECT id, option_text, essentials_id
             FROM question_options 
             WHERE question_id = ? AND deleted_at IS NULL
             ORDER BY id ASC`,
            [questionId]
        );

        const [[{ totalQuestions }]] = await pool.execute(
            `SELECT COUNT(*) AS totalQuestions 
             FROM questions 
             WHERE deleted_at IS NULL`
        );

        const essentialsId = options[0]?.essentials_id || null;
        let selectedValue = null;
        if (essentialsId && userId) {
            const [response] = await pool.execute(
                `SELECT value 
                 FROM essentials_responses 
                 WHERE user_id = ? AND essentials_id = ? 
                 ORDER BY updated_at DESC 
                 LIMIT 1`,
                [userId, essentialsId]
            );
            if (response.length > 0) {
                selectedValue = response[0].value;
            }
        }

        const formattedOptions = options.map(opt => {
            let text = opt.option_text;
            if (opt.essentials_id === 7 && selectedValue) {
                text = selectedValue;
            }
            return {
                id: opt.id,
                essentials_id: opt.essentials_id,
                text,
                is_selected: opt.essentials_id === 7 ? !!text : selectedValue === opt.option_text
            };
        });

        return {
            id: questions[0].id,
            question: questions[0].question,
            warning_heading: questions[0].warning_heading,
            warrning: questions[0].warrning,
            totalQuestions,
            options: formattedOptions
        };

    } catch (error) {
        console.error("Error in getQuestionWithOptions model:", error);
        throw error;
    }
};

export const getOptionsByEssentialsId = async (essentialsId) => {
    try {
        const [rows] = await pool.execute(
            `SELECT id, option_text as text, essentials_id
             FROM question_options 
             WHERE essentials_id = ? AND deleted_at IS NULL`,
            [essentialsId]
        );
        return rows;
    } catch (error) {
        console.error("DB error in getOptionsByEssentialsId:", error);
        throw error;
    }
};

export const getLatestEssentialValue = async (user_id, essentials_id) => {
    const [rows] = await pool.execute(
        `SELECT id, value 
         FROM essentials_responses 
         WHERE user_id = ? AND essentials_id = ?
         ORDER BY updated_at DESC 
         LIMIT 1`,
        [user_id, essentials_id]
    );
    return rows.length > 0 ? { id: rows[0].id, value: rows[0].value } : null;
};

export const updateEssentialScore = async (response_id, user_id, score) => {
    await pool.execute(
        `UPDATE essentials_responses
         SET score = ?
         WHERE id = ? AND user_id = ?`,
        [score, response_id, user_id]
    );
};

export const saveUserEssential = async (user_id, date, day, essentials_id, value, score) => {
    const startOfDay = Math.floor(new Date(new Date(date * 1000).setHours(0, 0, 0, 0)).getTime() / 1000);
    const endOfDay = Math.floor(new Date(new Date(date * 1000).setHours(23, 59, 59, 999)).getTime() / 1000);
    try {
        const [existing] = await pool.execute(
            `SELECT id FROM essentials_responses
             WHERE user_id = ?
             AND essentials_id = ?
             AND date BETWEEN ? AND ?
             LIMIT 1`,
            [user_id, essentials_id, startOfDay, endOfDay]
        );
        if (existing.length > 0) {
            await pool.execute(
                `UPDATE essentials_responses
                 SET value = ?, score = ?, updated_at = NOW()
                 WHERE id = ?`,
                [value, score, existing[0].id]
            );
        } else {
            await pool.execute(
                `INSERT INTO essentials_responses 
                 (user_id, date, day, essentials_id, value, score, created_at, updated_at)
                 VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())`,
                [user_id, date, day, essentials_id, value, score]
            );
        }
    } catch (error) {
        console.error("Error in saveUserEssential model:", error);
        throw error;
    }
};

export const getLatestQuote = async (startOfDay, endOfDay) => {
    try {
        const [rows] = await pool.execute(
            `
      SELECT id, quote_text, author_name, created_at, updated_at
      FROM quotes
      WHERE date BETWEEN ? AND ?
        AND deleted_at IS NULL
      ORDER BY created_at DESC
      LIMIT 1
      `,
            [startOfDay, endOfDay]
        );
        if (rows.length > 0) return rows[0];
        const [randomRows] = await pool.execute(
            `
      SELECT id, quote_text, author_name, created_at, updated_at
      FROM quotes
      WHERE deleted_at IS NULL
      ORDER BY RAND()
      LIMIT 1
      `
        );
        return randomRows[0] || null;
    } catch (error) {
        console.error("❌ Error fetching latest quote:", error);
        throw new Error("Database query failed");
    }
};

// export const getEssentials = async (user_id, startOfDay, endOfDay, day) => {
//     try {
//         const [essentials] = await pool.execute(
//             `SELECT 
//                 id AS essentials_id, 
//                 essential_name, 
//                 essential_image, 
//                 essential_description, 
//                 total_score 
//              FROM essentials
//              WHERE deleted_at IS NULL
//              ORDER BY id ASC`
//         );

//         const [responses] = await pool.execute(
//             `SELECT essentials_id, value, score
//              FROM essentials_responses
//              WHERE user_id = ? AND date BETWEEN ? AND ? AND day = ? AND deleted_at IS NULL`,
//             [user_id, startOfDay, endOfDay, day]
//         );

//         const essential6Response = responses.find(r => r.essentials_id === 6);
//         const result = essentials.map(e => {
//             const resp = responses.find(r => r.essentials_id === e.essentials_id);
//             if (e.essentials_id === 5) {
//                 return {
//                     ...e,
//                     essential_image: e.essential_image ? formatUrl(e.essential_image) : null,
//                     // value: resp?.value ?? null,
//                     // score: resp?.score ?? null,
//                     value: essential6Response?.value ?? null,
//                     score: formatScore(essential6Response?.score ?? null)
//                 };
//             }
//             return {
//                 ...e,
//                 essential_image: e.essential_image ? formatUrl(e.essential_image) : null,
//                 value: resp?.value ?? null,
//                 score: formatScore(resp?.score ?? null)
//             };
//         });
//         return result;

//     } catch (error) {
//         console.error("Error in getEssentials:", error);
//         throw error;
//     }
// };

export const getEssentials = async (user_id, startOfDay, endOfDay, day) => {
    try {
        const [essentials] = await pool.execute(
            `SELECT 
                id AS essentials_id, 
                essential_name, 
                essential_image, 
                essential_description, 
                total_score 
             FROM essentials
             WHERE deleted_at IS NULL
             ORDER BY id ASC`
        );

        const [responses] = await pool.execute(
            `SELECT essentials_id, value, score
             FROM essentials_responses
             WHERE user_id = ? AND date BETWEEN ? AND ? AND day = ? AND deleted_at IS NULL`,
            [user_id, startOfDay, endOfDay, day]
        );

        // const socialMapping = {
        //     "0 (No social interaction)": "0 Person",
        //     "1 (Interacted with 1 person)": "1 Person",
        //     "2 (Interacted with 2 person)": "2 Persons",
        //     "2+ (Interacted with two or more people)": "2+ Person"
        // };

        const essential6Response = responses.find(r => r.essentials_id === 6);
        const result = essentials.map(e => {
            const resp = responses.find(r => r.essentials_id === e.essentials_id);
            let value = resp?.value ?? null;
            // if (e.essentials_id === 4 && value && socialMapping[value]) {
            //     value = socialMapping[value];
            // }
            if (e.essentials_id === 4 && value) {
                let mappedValue = null;
                if (value.startsWith("0")) mappedValue = "0 Person";
                else if (value.startsWith("1")) mappedValue = "1 Person";
                else if (value.startsWith("2+")) mappedValue = "2+ Person";
                else if (value.startsWith("2")) mappedValue = "2 Persons";
                value = mappedValue;
            }
            if (e.essentials_id === 7 && value) {
                value = `${value} OZ`;
            }
            if (e.essentials_id === 5) {
                value = essential6Response?.value ?? null;
                return {
                    ...e,
                    essential_image: e.essential_image ? formatUrl(e.essential_image) : null,
                    value,
                    score: formatScore(essential6Response?.score ?? null)
                };
            }
            return {
                ...e,
                essential_image: e.essential_image ? formatUrl(e.essential_image) : null,
                value,
                score: formatScore(resp?.score ?? null)
            };
        });
        return result;
    } catch (error) {
        console.error("Error in getEssentials:", error);
        throw error;
    }
};

export const getNotifications = async (user_id, page = 1, limit = 10) => {
    const offset = (page - 1) * limit;
    const [rows] = await pool.query(
        `SELECT id, user_id, title, message, pushType, created_at, updated_at
         FROM ${Tables.NOTIFICATIONS}
         WHERE deleted_at IS NULL AND user_id = ?
         ORDER BY id DESC
         LIMIT ? OFFSET ?`,
        [user_id, limit, offset]
    );
    const [[{ total }]] = await pool.query(
        `SELECT COUNT(*) as total
         FROM ${Tables.NOTIFICATIONS}
         WHERE deleted_at IS NULL AND user_id = ?`,
        [user_id]
    );
    return { data: rows, total };
};

export const deleteNotification = async (user_id, id) => {
    const query = `
        UPDATE ${Tables.NOTIFICATIONS}
        SET deleted_at = NOW()
        WHERE id = ? AND user_id = ? AND deleted_at IS NULL
    `;
    const [result] = await pool.query(query, [id, user_id]);
    return result;
};

export const essentialsTips = async (essentialsId, page, limit) => {
    try {
        const offset = (page - 1) * limit;
        const [rows] = await pool.execute(
            `SELECT id, essentials_id, essentials_tips 
             FROM essentials_tips 
             WHERE essentials_id = ? AND deleted_at IS NULL
             ORDER BY id DESC
             LIMIT ? OFFSET ?`,
            [essentialsId, limit, offset]
        );
        const [[{ total }]] = await pool.execute(
            `SELECT COUNT(*) as total 
             FROM essentials_tips 
             WHERE essentials_id = ? AND deleted_at IS NULL`,
            [essentialsId]
        );
        return { data: rows, total };
    } catch (error) {
        console.error("DB error in essentialsTips model:", error);
        throw error;
    }
};

export const insertNotifications = async (notifications = [], batchSize = 500) => {
    if (!notifications.length) return { affectedRows: 0 };
    const now = new Date();
    try {
        for (let i = 0; i < notifications.length; i += batchSize) {
            const batch = notifications.slice(i, i + batchSize);
            const flatValues = [];
            const placeholders = batch.map(n => {
                flatValues.push(n.user_id, n.title ?? "", n.message ?? "", n.pushType ?? "0", now, now);
                return "(?, ?, ?, ?, ?, ?)";
            }).join(", ");
            const query = `
        INSERT INTO ${Tables.NOTIFICATIONS} 
        (user_id, title, message, pushType, created_at, updated_at)
        VALUES ${placeholders}
      `;
            await pool.execute(query, flatValues);
        }
        return { affectedRows: notifications.length };
    } catch (error) {
        console.error("Error in insertNotifications:", error);
        throw error;
    }
};

export const getUsersWithoutSubscription = async () => {
    const query = `
    SELECT u.id, u.user_name, u.device_type, u.device_token
    FROM users u
    LEFT JOIN subscriptions s 
      ON u.id = s.user_id AND s.status = 'active'
    WHERE s.id IS NULL
      AND u.is_logged_in = 1
      AND u.is_deleted = 0
      AND u.device_token IS NOT NULL
  `;
    const [rows] = await pool.query(query);
    return rows;
};

export const getUsersWithIncompleteEssentials = async (startOfDay, endOfDay) => {
    const allowedEssentials = [1, 2, 3, 4, 5, 7, 8, 9, 10, 11];
    const query = `
    SELECT 
      u.id, 
      u.user_name, 
      u.device_type, 
      u.device_token
    FROM users u
    INNER JOIN subscriptions s
      ON u.id = s.user_id 
      AND s.status = 'active'
    LEFT JOIN essentials_responses r 
      ON u.id = r.user_id 
      AND r.date BETWEEN ? AND ? 
      AND r.essentials_id IN (${allowedEssentials.join(',')})
      AND r.deleted_at IS NULL
    WHERE
      u.is_deleted = 0
      AND u.is_logged_in = 1
      AND u.device_token IS NOT NULL
    GROUP BY u.id
    HAVING COUNT(DISTINCT r.essentials_id) < ?
  `;
    const [rows] = await pool.query(query, [startOfDay, endOfDay, allowedEssentials.length]);
    return rows;
};

export const fetchUserMonths = async (user_id) => {
    try {
        const [rows] = await pool.execute(
            `
            SELECT DISTINCT r.date
            FROM essentials_responses r
            WHERE r.user_id = ?
            ORDER BY r.date ASC
            `,
            [user_id]
        );
        const monthsSet = new Set();
        const result = [];
        rows.forEach(row => {
            if (!row.date) return;
            const d = new Date(row.date * 1000);
            const year = d.getUTCFullYear();
            const month = d.getUTCMonth();
            const monthStart = Date.UTC(year, month, 1) / 1000;
            if (!monthsSet.has(monthStart)) {
                monthsSet.add(monthStart);
                result.push({
                    month_epoch: monthStart,
                    month: `${year}-${String(month + 1).padStart(2, '0')}`
                });
            }
        });
        return result;
    } catch (error) {
        console.error("Error in fetchUserMonths model:", error);
        throw error;
    }
};

export const getUserEssentialsWithScores = async (user_id) => {
    try {
        const [rows] = await pool.execute(
            `SELECT
          e.id AS essentials_id,
          e.essential_name AS name,
          e.essential_image AS image,
          e.essential_description AS description,
          e.total_score,
          r.score,
          r.date
       FROM ${Tables.ESSENTIALS} e
       LEFT JOIN essentials_responses r
          ON e.id = r.essentials_id
          AND r.user_id = ?
       WHERE e.deleted_at IS NULL
       ORDER BY r.date ASC`,
            [user_id]
        );
        return rows.map(row => ({
            ...row,
            image: row.image ? formatUrl(row.image) : null
        }));
    } catch (error) {
        console.error("Error in getUserEssentialsWithScores model:", error);
        throw error;
    }
};