import pool from "../config/database.js";
import { Tables } from "../utils/dbTables.js";
import { formatUrl } from "../utils/commonFunctions.js";


export const getUserByAccessToken = async (access_token) => {
    try {
        const query = `SELECT * FROM ${Tables.USERS} WHERE access_token = ? LIMIT 1`;
        const [rows] = await pool.execute(query, [access_token]);
        return rows.length ? rows[0] : null;
    } catch (error) {
        console.error("Error in getUserByAccessToken:", error);
        throw new Error("Database query failed");
    }
};

export const checkUserEmail = async (email) => {
    try {
        const query = `
      SELECT * FROM ${Tables.USERS}
      WHERE email = ? AND is_deleted = ? 
      LIMIT 1
    `;
        const [rows] = await pool.execute(query, [email, 0]);
        return rows.length > 0 ? rows[0] : null;
    } catch (error) {
        console.error("Error in checkUserEmail:", error);
        throw new Error("Database query failed");
    }
};

export const signUp = async (body) => {
    try {
        const query = `
            INSERT INTO ${Tables.USERS} (user_name, email, password, profile_image, time_zone) 
            VALUES (?, ?, ?, ?, ?)
        `;
        const [result] = await pool.execute(query, [
            body.user_name || null,
            body.email || null,
            body.password || null,
            body.profile_image || null,
            body.time_zone || null
        ]);
        return result;
    } catch (error) {
        console.error("Error in signUp:", error.message);
        throw new Error("Database insertion failed");
    }
};

export const getUserById = async (user_id) => {
    try {
        const query = `SELECT * FROM ${Tables.USERS} WHERE id = ? LIMIT 1`;
        const [rows] = await pool.execute(query, [user_id]);
        if (!rows.length) return null;
        const user = rows[0];
        return {
            ...user,
            profile_image: user.profile_image ? formatUrl(user.profile_image) : null,
        };
    } catch (error) {
        console.error("Error in getUserById:", error.message);
        throw new Error("Database query failed");
    }
};

export const updateUsersTable = async (access_token, body_data, userId) => {
    try {
        const query = `
      UPDATE ${Tables.USERS} 
      SET 
        access_token = ?,
        device_token = ?,
        device_type = ?,
        google_token = ?,
        is_logged_in = 1,
        is_email_verified = 1,
        time_zone = ?
      WHERE id = ?
    `;
        const params = [
            access_token,
            body_data.device_token || null,
            body_data.device_type || null,
            body_data.google_token || null,
            body_data.time_zone || null,
            userId,
        ];
        const [result] = await pool.execute(query, params);
        return result;
    } catch (error) {
        console.error("Error in updateUsersTable:", error.message);
        throw new Error("Database update failed");
    }
};

export const updatePassword = async (id, password) => {
    try {
        const query = `UPDATE ${Tables.USERS} SET password = ? WHERE id = ?`;
        const [result] = await pool.execute(query, [password, id]);
        return result;
    } catch (err) {
        console.error("Error in updatePassword:", err.message);
        throw new Error("Database update failed");
    }
};

export const updateUsersTableInGoogleLogin = async (access_token, body_data, userId) => {
    try {
        const query = `
      UPDATE ${Tables.USERS} 
      SET 
        access_token = ?,
        device_token = ?,
        device_type = ?,
        google_token = ?,
        is_logged_in = 1,
        is_email_verified = 1,
        time_zone = ?
      WHERE id = ?
    `;
        const params = [
            access_token,
            body_data.device_token || null,
            body_data.device_type || null,
            body_data.google_token || null,
            body_data.time_zone || null,
            userId,
        ];
        const [result] = await pool.execute(query, params);
        console.log("✅ Google login update successful:", result);
        return result;
    } catch (error) {
        console.error("❌ Error updating user in Google login:", error.message);
        throw new Error("Database update failed");
    }
};

export const getUserByAppleToken = async (apple_token) => {
    try {
        const query = `SELECT * FROM ${Tables.USERS} WHERE apple_token = ? AND is_deleted = 0 LIMIT 1`;
        const [rows] = await pool.execute(query, [apple_token]);
        return rows[0] || null;
    } catch (error) {
        console.error("Error in getUserByAppleToken:", error.message);
        throw error;
    }
};

export const updateAppleToken = async (userId, apple_token) => {
    try {
        const query = `
      UPDATE ${Tables.USERS}
      SET apple_token = ?
      WHERE id = ?
    `;
        const params = [apple_token || null, userId];
        const [result] = await pool.execute(query, params);
        return result;
    } catch (error) {
        console.error("Error in updateAppleToken:", error.message);
        throw error;
    }
};

export const updateAppleUser = async (userId, updates) => {
    try {
        const query = `
      UPDATE ${Tables.USERS} SET 
        access_token = ?, 
        device_token = ?, 
        device_type = ?, 
        is_logged_in = 1, 
        is_email_verified = 1,
        time_zone = ?
      WHERE id = ?
    `;
        const params = [
            updates.access_token,
            updates.device_token || null,
            updates.device_type || null,
            updates.time_zone || null,
            userId,
        ];
        const [result] = await pool.execute(query, params);
        return result;
    } catch (error) {
        console.error("Error in updateAppleUser:", error.message);
        throw error;
    }
};

export const getDeletedAppleUser = async (apple_token) => {
    try {
        const query = `SELECT * FROM ${Tables.USERS} WHERE apple_token = ? AND is_deleted = 1 LIMIT 1`;
        const [rows] = await pool.execute(query, [apple_token]);
        return rows[0] || null;
    } catch (error) {
        console.error("Error in getDeletedAppleUser:", error.message);
        throw error;
    }
};

export const createAppleUser = async ({ email, user_name, apple_token, device_token, device_type, profile_image }) => {
    try {
        const query = `
      INSERT INTO ${Tables.USERS} 
        (email, user_name, apple_token, device_token, device_type, profile_image, is_email_verified, is_logged_in ) 
      VALUES (?, ?, ?, ?, ?, ?, 1, 1 )
    `;
        const [result] = await pool.execute(query, [
            email,
            user_name,
            apple_token,
            device_token || null,
            device_type || null,
            profile_image || null
        ]);
        return result;
    } catch (error) {
        console.error("Error in createAppleUser:", error.message);
        throw error;
    }
};

export const verifyEmail = async (userId) => {
    try {
        const query = `UPDATE ${Tables.USERS} SET is_email_verified = 1 WHERE id = ?`;
        const [result] = await pool.execute(query, [userId]);
        return result.affectedRows > 0;
    } catch (error) {
        console.error("Error in verifyEmail:", error.message);
        throw new Error("Database update failed");
    }
};

export const saveHelpAndSupport = async ({ user_id, user_name, email, message }) => {
    try {
        const query = `
      INSERT INTO ${Tables.HELP_SUPPORT} (user_id, user_name, email, message, created_at)
      VALUES (?, ?, ?, ?, NOW())
    `;
        const [result] = await pool.execute(query, [
            user_id ?? null,
            user_name ?? null,
            email ?? null,
            message ?? null
        ]);
        return result.insertId;
    } catch (error) {
        console.error("Error in saveHelpAndSupport:", error.message);
        throw new Error("Database query failed");
    }
};

export const checkUserById = async (user_id) => {
    try {
        const query = `SELECT * FROM ${Tables.USERS} WHERE id = ? AND is_deleted = 0 LIMIT 1`;
        const [rows] = await pool.execute(query, [user_id]);
        return rows.length > 0 ? rows[0] : null;
    } catch (error) {
        console.error("Error in checkUserById:", error.message);
        throw new Error("Database query failed");
    }
};

export const logout = async (userId) => {
    try {
        const query = `
      UPDATE ${Tables.USERS} 
      SET is_logged_in = 0, device_type = NULL, device_token = NULL, access_token = NULL 
      WHERE id = ?
    `;
        const [result] = await pool.execute(query, [userId]);
        return result;
    } catch (error) {
        console.error("Logout Model Error:", error);
        throw error;
    }
};

export const saveAccountDeletionOtp = async (userId, otp) => {
    try {
        await pool.execute(
            `UPDATE ${Tables.USERS} 
             SET account_delete_otp = ?  
             WHERE id = ? AND is_deleted = 0`,
            [otp, userId]
        );
    } catch (error) {
        console.error('saveAccountDeletionOtp Error:', error);
        throw error;
    }
};

export const deleteAccount = async (userId) => {
    try {
        const query = `
      UPDATE ${Tables.USERS} 
      SET account_delete_otp = NULL, is_logged_in = 0, is_deleted = 1, device_type = null , device_token = null  where id=?`;
        const [result] = await pool.execute(query, [userId]);
        return result;
    } catch (error) {
        console.error("Logout Model Error:", error);
        throw error;
    }
};