import pool from "../config/database.js";
import { Tables } from "../utils/dbTables.js";


export const storeGoogleTokens = async (access_token, refresh_token, access_token_expires_at, refresh_token_expires_at) => {
    try {
        const [result] = await pool.execute(
            `INSERT INTO ${Tables.GOOGLE_TOKENS} (
         id, access_token, refresh_token, access_token_expires_at, refresh_token_expires_at
       )
       VALUES (1, ?, ?, ?, ?)
       ON DUPLICATE KEY UPDATE
         access_token = VALUES(access_token),
         refresh_token = VALUES(refresh_token),
         access_token_expires_at = VALUES(access_token_expires_at),
         refresh_token_expires_at = VALUES(refresh_token_expires_at),
         updated_at = CURRENT_TIMESTAMP`,
            [access_token, refresh_token, access_token_expires_at, refresh_token_expires_at]
        );
        return result;
    } catch (error) {
        console.error("Error storing Google tokens:", error);
        throw error;
    }
};

export const getActiveSubscription = async (user_id) => {
    try {
        const [rows] = await pool.execute(
            `SELECT *
       FROM ${Tables.SUBSCRIPTIONS}
       WHERE user_id = ? AND status = ?
       ORDER BY created_at DESC
       LIMIT 1`,
            [user_id, "active"]
        );
        return rows[0] || null;
    } catch (error) {
        console.error("Error fetching latest active subscription:", error);
        throw error;
    }
};

export const insertSubscription = async ({
    user_id,
    plan_id,
    subscription_id,
    expires_at,
    plan_price,
    product_id,
    device_type,
    package_name,
    order_id,
    purchase_token,
    transaction_id,
    plan_duration,
    status,
    event_type,
}) => {
    try {
        const [result] = await pool.execute(
            `INSERT INTO ${Tables.SUBSCRIPTIONS} (
                user_id, plan_id, subscription_id, expires_at, plan_price, product_id,
                device_type, package_name, order_id, purchase_token, transaction_id,
                plan_duration, status, event_type
            )
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [
                user_id,
                plan_id,
                subscription_id,
                expires_at,
                plan_price,
                product_id,
                device_type,
                package_name,
                order_id,
                purchase_token,
                transaction_id,
                plan_duration,
                status,
                event_type,
            ]
        );
        return result;
    } catch (error) {
        console.error("Error inserting subscription:", error);
        throw error;
    }
};

export const updateSubscriptionStatus = async (user_id, id, status, event_type) => {
    try {
        const [result] = await pool.execute(
            `UPDATE ${Tables.SUBSCRIPTIONS}
       SET status = ?, event_type = ?, updated_at = CURRENT_TIMESTAMP
       WHERE user_id = ? AND id = ? AND status = 'active'`,
            [status, event_type, user_id, id]
        );
        return result;
    } catch (error) {
        console.error("Error updating subscription status:", error);
        throw error;
    }
};

export const getGoogleTokens = async () => {
    try {
        const [rows] = await pool.execute(
            `SELECT access_token, refresh_token, access_token_expires_at, refresh_token_expires_at
       FROM ${Tables.GOOGLE_TOKENS}
       WHERE id = 1
       LIMIT 1`
        );
        return rows;
    } catch (error) {
        console.error("Error fetching Google tokens:", error);
        throw error;
    }
};

export const getActiveAndroidSubscriptions = async () => {
    try {
        const [rows] = await pool.execute(
            `SELECT id, user_id, subscription_id, plan_id, plan_price, plan_duration, 
              expires_at, product_id, device_type, package_name, 
              order_id, purchase_token, transaction_id 
       FROM ${Tables.SUBSCRIPTIONS}
       WHERE status = 'active' AND device_type = 'Android'
       ORDER BY created_at ASC`
        );
        return rows;
    } catch (error) {
        console.error("Error fetching active Android subscriptions:", error);
        throw error;
    }
};