import pool from "../config/database.js";
import { Tables } from "../utils/dbTables.js";
import { formatUrl } from "../utils/commonFunctions.js";


export const insertMassPushHistory = async (summary) => {
  const { title, message, totalUsers, successCount, failureCount, androidSent, iosSent } = summary;
  try {
    const query = `
      INSERT INTO mass_push_history 
      (title, message, total_users, success_count, failure_count, android_sent, ios_sent, created_at)
      VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
    `;
    const [result] = await pool.execute(query, [
      title,
      message,
      totalUsers,
      successCount,
      failureCount,
      androidSent,
      iosSent,
    ]);
    return result.insertId;
  } catch (error) {
    console.error("Error inserting mass push history:", error);
    throw error;
  }
};

export const getAllUsersWithDeviceTokens = async () => {
  try {
    const query = `
      SELECT id, device_type, device_token
      FROM ${Tables.USERS}
      WHERE is_deleted = 0
        AND is_blocked_by_admin = 0
        AND device_token IS NOT NULL
        AND device_type IN ('Android', 'Apple')
    `;
    const [rows] = await pool.execute(query);
    return rows;
  } catch (error) {
    console.error("❌ Error fetching users with device tokens:", error.message);
    throw error;
  }
};

// export const getUsersList = async (page = 1, limit = 10, search = "") => {
//   try {
//     const offset = (page - 1) * limit;
//     const searchParam = search.trim() ? `%${search.trim()}%` : null;
//     const query = `
//       SELECT 
//           id, 
//           user_name,
//           email,
//           is_blocked_by_admin,
//           gender, 
//           age, 
//           profile_image,
//           COUNT(*) OVER() AS total_count
//       FROM users
//       WHERE is_deleted = 0
//       ${searchParam ? 'AND (user_name LIKE ? OR email LIKE ?)' : ''}
//       ORDER BY id DESC
//       LIMIT ? OFFSET ?
//     `;
//     const params = searchParam ? [searchParam, searchParam, limit, offset] : [limit, offset];
//     const [rows] = await pool.execute(query, params);
//     const total = rows.length ? rows[0].total_count : 0;
//     const data = rows.map(row => ({
//       id: row.id,
//       user_name: row.user_name,
//       email: row.email,
//       is_blocked_by_admin: row.is_blocked_by_admin,
//       gender: row.gender,
//       age: row.age,
//       profile_image: row.profile_image ? formatUrl(row.profile_image) : null
//     }));
//     return { data, total };
//   } catch (error) {
//     console.error("Error in getUsersList model:", error);
//     throw error;
//   }
// };

export const getUsersList = async (page = 1, limit = 10, search = "") => {
  try {
    const offset = (page - 1) * limit;
    const searchParam = search.trim() ? `%${search.trim()}%` : null;
    const query = `
      SELECT 
        u.id, 
        u.user_name,
        u.email,
        u.is_blocked_by_admin,
        u.gender, 
        u.age, 
        u.profile_image,
        CASE 
          WHEN EXISTS (
            SELECT 1 
            FROM ${Tables.SUBSCRIPTIONS} s
            WHERE s.user_id = u.id
              AND s.status = 'active'
          ) THEN 1
          ELSE 0
        END AS is_subscribed,
        COUNT(*) OVER() AS total_count
      FROM ${Tables.USERS} u
      WHERE u.is_deleted = 0
      ${searchParam ? 'AND (u.user_name LIKE ? OR u.email LIKE ?)' : ''}
      ORDER BY u.id DESC
      LIMIT ? OFFSET ?
    `;
    const params = searchParam ? [searchParam, searchParam, limit, offset] : [limit, offset];
    const [rows] = await pool.execute(query, params);
    const total = rows.length ? rows[0].total_count : 0;
    const data = rows.map(row => ({
      id: row.id,
      user_name: row.user_name,
      email: row.email,
      is_blocked_by_admin: row.is_blocked_by_admin,
      gender: row.gender,
      age: row.age,
      profile_image: row.profile_image ? formatUrl(row.profile_image) : null,
      is_subscribed: !!row.is_subscribed,
    }));
    return { data, total };
  } catch (error) {
    console.error("Error in getUsersList model:", error);
    throw error;
  }
};

export const updateUserStatus = async (id, status) => {
  try {
    const query = `
      UPDATE users
      SET is_blocked_by_admin = ?, updated_at = NOW()
      WHERE id = ? AND is_deleted = 0
    `;
    const [result] = await pool.execute(query, [status, id]);
    return result;
  } catch (error) {
    console.error("Error in updateUserStatus model:", error);
    throw error;
  }
};

// export const getSubscribedUsersList = async (page = 1, limit = 10, search = "") => {
//   try {
//     const offset = (page - 1) * limit;
//     const searchTerm = search.trim();
//     const params = [];

//     let baseQuery = `
//       FROM users u
//       INNER JOIN ${Tables.SUBSCRIPTIONS} s 
//         ON u.id = s.user_id
//       WHERE u.is_deleted = 0
//         AND s.status = 'active'
//         AND s.id = (
//           SELECT MAX(ss.id)
//           FROM ${Tables.SUBSCRIPTIONS} ss
//           WHERE ss.user_id = u.id
//             AND ss.status = 'active'
//         )
//     `;

//     if (searchTerm) {
//       baseQuery += ` AND (u.user_name LIKE ? OR u.email LIKE ?)`;
//       params.push(`%${searchTerm}%`, `%${searchTerm}%`);
//     }

//     const dataQuery = `
//       SELECT 
//         u.id, 
//         u.user_name, 
//         u.email,
//         u.gender, 
//         u.age, 
//         u.profile_image,
//         s.expires_at,
//         s.plan_price,
//         s.plan_duration,
//         s.status,
//         s.event_type,
//         s.device_type
//       ${baseQuery}
//       ORDER BY u.id DESC
//       LIMIT ? OFFSET ?
//     `;
//     params.push(limit, offset);
//     const [rows] = await pool.execute(dataQuery, params);
//     const formattedData = rows.map(row => ({
//       ...row,
//       profile_image: row.profile_image ? formatUrl(row.profile_image) : null
//     }));

//     const countParams = searchTerm ? [`%${searchTerm}%`, `%${searchTerm}%`] : [];
//     const countQuery = `SELECT COUNT(DISTINCT u.id) as total ${baseQuery}`;
//     const [countRows] = await pool.execute(countQuery, countParams);
//     const total = countRows[0]?.total || 0;

//     return { data: formattedData, total };
//   } catch (error) {
//     console.error("Error in getSubscribedUsersList model:", error);
//     throw error;
//   }
// };

export const getUserDetailsById = async (userId) => {
  try {
    const query = `
      SELECT 
        u.id AS user_id,
        u.user_name,
        u.email,
        u.gender,
        u.age,
        u.profile_image,
        u.is_blocked_by_admin,
        u.created_at,
        u.updated_at,
        s.id AS subscription_id,
        s.plan_duration,
        s.plan_price,
        s.status AS subscription_status,
        s.device_type,
        s.event_type,
        s.expires_at,
        s.created_at AS subscribed_at
      FROM ${Tables.USERS} u
      LEFT JOIN ${Tables.SUBSCRIPTIONS} s
        ON s.id = (
          SELECT MAX(s2.id)
          FROM ${Tables.SUBSCRIPTIONS} s2
          WHERE s2.user_id = u.id
            AND s2.status = 'active'
        )
      WHERE u.id = ? AND u.is_deleted = 0
    `;
    const [rows] = await pool.execute(query, [userId]);
    if (!rows.length) return null;
    const user = {
      id: rows[0].user_id,
      user_name: rows[0].user_name,
      email: rows[0].email,
      gender: rows[0].gender,
      age: rows[0].age,
      profile_image: rows[0].profile_image ? formatUrl(rows[0].profile_image) : null,
      is_blocked_by_admin: rows[0].is_blocked_by_admin,
      created_at: rows[0].created_at,
      updated_at: rows[0].updated_at,
      subscriptions: [],
    };
    if (rows[0].subscription_id) {
      user.subscriptions.push({
        subscription_id: rows[0].subscription_id,
        plan_price: rows[0].plan_price ? Number(rows[0].plan_price) : 0,
        plan_duration: rows[0].plan_duration,
        status: rows[0].subscription_status,
        device_type: rows[0].device_type,
        event_type: rows[0].event_type,
        expires_at: rows[0].expires_at,
        subscribed_at: rows[0].subscribed_at
      });
    }
    return user;
  } catch (error) {
    console.error("Error in getUserDetailsById model:", error);
    throw error;
  }
};

function getDayRangeUTC(input) {
  let day;
  if (typeof input === 'number') {
    day = new Date(input * 1000);
  } else if (typeof input === 'string') {
    day = new Date(input + "T00:00:00Z");
  } else {
    throw new Error("Invalid input: must be a timestamp or date string");
  }
  const year = day.getUTCFullYear();
  const month = day.getUTCMonth();
  const date = day.getUTCDate();
  const startOfDay = Math.floor(Date.UTC(year, month, date, 0, 0, 0, 0) / 1000);
  const endOfDay = Math.floor(Date.UTC(year, month, date, 23, 59, 59, 999) / 1000);
  return { startOfDay, endOfDay };
}

export const insertQuote = async ({ quote_text, author_name, date }) => {
  try {
    const { startOfDay, endOfDay } = getDayRangeUTC(date);
    const [existing] = await pool.execute(
      `SELECT id FROM ${Tables.QUOTES} WHERE date BETWEEN ? AND ? AND deleted_at IS NULL LIMIT 1`,
      [startOfDay, endOfDay]
    );
    if (existing.length > 0) {
      return { exists: true };
    }
    const [result] = await pool.execute(
      `INSERT INTO ${Tables.QUOTES} (quote_text, author_name, date) VALUES (?, ?, ?)`,
      [quote_text.trim(), author_name.trim(), date]
    );
    return { insertId: result.insertId, exists: false };
  } catch (error) {
    console.error("Error in insertQuote model:", error);
    throw error;
  }
};

export const updateQuote = async ({ quote_id, quote_text, author_name, date }) => {
  try {
    const { startOfDay, endOfDay } = getDayRangeUTC(date);
    const [existing] = await pool.execute(
      `
      SELECT id FROM ${Tables.QUOTES}
      WHERE date BETWEEN ? AND ?
        AND id != ?
        AND deleted_at IS NULL
      LIMIT 1
      `,
      [startOfDay, endOfDay, quote_id]
    );
    if (existing.length > 0) {
      return { exists: true };
    }
    const [result] = await pool.execute(
      `
      UPDATE ${Tables.QUOTES}
      SET quote_text = ?, author_name = ?, date = ?, updated_at = NOW()
      WHERE id = ? AND deleted_at IS NULL
      `,
      [quote_text.trim(), author_name.trim(), date, quote_id]
    );
    return { updated: result.affectedRows > 0, exists: false };
  } catch (error) {
    console.error("Error in updateQuote model:", error);
    throw error;
  }
};

export const softDeleteQuote = async (quoteId) => {
  try {
    const query = `
      UPDATE ${Tables.QUOTES} 
      SET deleted_at = NOW() 
      WHERE id = ? AND deleted_at IS NULL
    `;
    const [result] = await pool.execute(query, [quoteId]);
    return result;
  } catch (error) {
    console.log("Error in softDeleteQuote model:", error);
    throw error;
  }
};

export const getQuotesList = async (page = 1, limit = 10, search = "") => {
  try {
    const offset = (page - 1) * limit;
    const searchParam = search ? `%${search}%` : null;
    let countQuery = `SELECT COUNT(*) as total FROM ${Tables.QUOTES} WHERE deleted_at IS NULL`;
    const countParams = [];
    if (searchParam) {
      countQuery += ` AND (quote_text LIKE ? OR author_name LIKE ?)`;
      countParams.push(searchParam, searchParam);
    }
    const [countRows] = await pool.execute(countQuery, countParams);
    const total = countRows[0].total;
    let dataQuery = `
      SELECT id, quote_text, author_name, date, created_at, updated_at
      FROM ${Tables.QUOTES}
      WHERE deleted_at IS NULL
    `;
    const params = [];
    if (searchParam) {
      dataQuery += ` AND (quote_text LIKE ? OR author_name LIKE ?)`;
      params.push(searchParam, searchParam);
    }
    dataQuery += ` ORDER BY date DESC LIMIT ? OFFSET ?`;
    params.push(limit, offset);
    const [rows] = await pool.execute(dataQuery, params);
    const data = rows.map(row => ({
      id: row.id,
      quote_text: row.quote_text,
      author_name: row.author_name,
      date: row.date,
      created_at: row.created_at,
      updated_at: row.updated_at,
    }));
    return { data, total };
  } catch (error) {
    console.error("Error in getQuotesList model:", error);
    throw error;
  }
};

export const bulkUpsertQuotes = async (rows) => {
  const insertedIds = [];
  const invalidRows = [];

  if (!rows.length) return { insertedIds, invalidRows };

  for (let i = 0; i < rows.length; i++) {
    const row = rows[i];
    const quote_text = (row.quote_text || "").trim();
    const author_name = (row.author_name || "").trim();
    const date_str = (row.date || "").trim();

    // Validate 
    const errors = [];
    if (!quote_text) errors.push("quote_text is required");
    if (!author_name) errors.push("author_name is required");
    if (!date_str) errors.push("date is required");
    if (errors.length > 0) {
      invalidRows.push({
        rowNumber: i + 2, // +2 because header is row 1
        errors,
      });
      continue;
    }

    const isoDateRegex = /^\d{4}-\d{2}-\d{2}$/;
    if (!isoDateRegex.test(date_str)) {
      invalidRows.push({
        rowNumber: i + 2,
        errors: [`Invalid date format, should be YYYY-MM-DD (${date_str})`],
      });
      continue;
    }

    try {
      const { startOfDay, endOfDay } = getDayRangeUTC(date_str);

      const [existing] = await pool.execute(
        `SELECT id FROM ${Tables.QUOTES} WHERE date BETWEEN ? AND ? AND deleted_at IS NULL LIMIT 1`,
        [startOfDay, endOfDay]
      );

      if (existing.length > 0) {
        invalidRows.push({
          rowNumber: i + 2,
          errors: [`Quote already exists for this date (${date_str})`],
        });
        continue;
      }

      const [insertResult] = await pool.execute(
        `INSERT INTO ${Tables.QUOTES} (quote_text, author_name, date) VALUES (?, ?, ?)`,
        [quote_text, author_name, startOfDay]
      );

      if (insertResult.affectedRows > 0)
        insertedIds.push(insertResult.insertId);

    } catch (err) {
      invalidRows.push({
        rowNumber: i + 2,
        errors: [`Invalid date format or database error for date (${date_str})`],
      });
    }
  }

  return { insertedIds, invalidRows };
};

export const getMassPushHistory = async ({ search = "", page = 1, limit = 10 }) => {
  try {
    const offset = (page - 1) * limit;
    const searchTerm = `%${search.trim()}%`;
    const countQuery = `
      SELECT COUNT(*) AS total
      FROM mass_push_history
      WHERE deleted_at IS NULL
        AND (? = '' OR title LIKE ? OR message LIKE ?)
    `;
    const [[{ total }]] = await pool.execute(countQuery, [search, searchTerm, searchTerm]);
    const dataQuery = `
      SELECT 
        id,
        title,
        message,
        total_users,
        success_count,
        failure_count,
        android_sent,
        ios_sent,
        created_at
      FROM mass_push_history
      WHERE deleted_at IS NULL
        AND (? = '' OR title LIKE ? OR message LIKE ?)
      ORDER BY created_at DESC
      LIMIT ? OFFSET ?
    `;
    const [rows] = await pool.execute(dataQuery, [search, searchTerm, searchTerm, limit, offset]);
    return { total, rows };
  } catch (error) {
    console.error("Error fetching mass push history:", error);
    throw error;
  }
};