import Joi from "joi";

const passwordPattern = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[!@#$%^&*()])[A-Za-z\d!@#$%^&*()]{8,}$/;

export const signup = Joi.object({
    user_name: Joi.string().min(2).max(50).required().messages({
        "string.min": "User name must be at least 2 characters",
        "string.max": "User name cannot exceed 50 characters",
        "any.required": "User name is required",
    }),
    email: Joi.string().email().required().messages({
        "string.email": "Please enter a valid email address",
        "any.required": "Email is required",
    }),
    password: Joi.string().pattern(passwordPattern).required().messages({
        "string.pattern.base":
            "Password must contain uppercase, lowercase, number & special character, minimum 6 characters",
        "any.required": "Password is required",
    }),
    confirm_password: Joi.string()
        .valid(Joi.ref("password"))
        .required()
        .messages({
            "any.only": "Confirm password must match password",
            "any.required": "Confirm password is required",
        }),
});

export const login = Joi.object({
    email: Joi.string().email().required().messages({
        "string.email": "Please enter a valid email",
        "any.required": "Email is required",
    }),
    password: Joi.string().required().messages({
        "string.pattern.base": "Password must contain uppercase, lowercase, number & special character",
        "any.required": "Password is required",
    }),
    device_token: Joi.string().optional().allow(""),
    device_type: Joi.string()
        .valid("Android", "Apple", "Web")
        .optional()
        .messages({
            "any.only": "Device type must be either Android, Apple, or Web",
        }),
    time_zone: Joi.string().optional().allow("").messages({
        "string.base": "Time zone must be a string",
    }),
});

export const forgotPassword = Joi.object({
    email: Joi.string().email().required().messages({
        "string.email": "Please enter a valid email",
        "any.required": "Email is required",
    }),
});

export const googleLogin = Joi.object({
    email: Joi.string().email().required().messages({
        "string.email": "Please enter a valid email",
        "any.required": "Email is required",
    }),
    user_name: Joi.string().min(2).max(50).required().messages({
        "string.min": "User name must be at least 2 characters",
        "string.max": "User name cannot exceed 50 characters",
        "any.required": "User name is required",
    }),
    device_token: Joi.string().optional().allow(""),
    device_type: Joi.string().valid("Android", "Apple", "Web").optional().messages({
        "any.only": "Device type must be either Android, Apple, or Web",
    }),
    google_token: Joi.string().optional().allow(""),
    profile_image: Joi.string().optional().allow(null, ""),
    time_zone: Joi.string().optional().allow("").messages({
        "string.base": "Time zone must be a string",
    }),
});

export const appleLogin = Joi.object({
    email: Joi.string()
        .email()
        .allow("", null)
        .messages({
            "string.email": "Email must be a valid email",
        }),
    device_token: Joi.string()
        .required()
        .messages({
            "string.empty": "Device token is required",
        }),
    device_type: Joi.string()
        .valid("Android", "Apple", "Web")
        .required()
        .messages({
            "any.only": "Device type must be Android, Apple, or Web",
            "string.empty": "Device type is required",
        }),
    apple_token: Joi.string()
        .required()
        .messages({
            "string.empty": "Apple token is required",
        }),
    user_name: Joi.string()
        .min(2)
        .max(50)
        .allow('')
        .messages({
            "string.min": "User name must be at least 2 characters",
            "string.max": "User name cannot exceed 50 characters",
        }),
    profile_image: Joi.string()
        .allow("", null)
        .messages({
            "string.base": "Profile image must be a string",
        }),
    time_zone: Joi.string().optional().allow("").messages({
        "string.base": "Time zone must be a string",
    }),
});

export const helpAndSupport = Joi.object({
    email: Joi.string()
        .trim()
        .lowercase()
        .email({ tlds: { allow: false } })
        .required()
        .messages({
            "string.base": "Email must be a string.",
            "string.empty": "Email is required.",
            "string.email": "Email must be a valid email address.",
            "any.required": "Email is required.",
        }),
    message: Joi.string()
        .trim()
        .min(3)
        .max(1000)
        .required()
        .messages({
            "string.base": "Message must be a string.",
            "string.empty": "Message is required.",
            "string.min": "Message must be at least 3 characters long.",
            "string.max": "Message must not exceed 1000 characters.",
            "any.required": "Message is required.",
        }),
});