import { errorMessage } from "../../../../utils/errorMessage.js";
import { successMessage } from "../../../../utils/successMessage.js";
import { formatUrl, calculateEssentialScore, fetchProgress, formatScore } from "../../../../utils/commonFunctions.js";
import * as userModel from "../../../../models/userModel.js";
import * as userAuthModel from "../../../../models/userAuthModel.js";


export const getUserProfile = async (req, res) => {
    try {
        const userId = req.user_id;
        const user = await userAuthModel.getUserById(userId);

        delete user.password;
        return res.status(successMessage.LOGGED_IN_USER_PROFILE.code).json({
            code: successMessage.LOGGED_IN_USER_PROFILE.code,
            message: successMessage.LOGGED_IN_USER_PROFILE.message,
            data: user,
        });

    } catch (error) {
        console.error("Error in getUserProfile:", error);
        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
        });
    }
};

export const markWelcomeDone = async (req, res) => {
    try {
        const userId = req.user_id;
        await userModel.updateUserById(userId, { is_welcome_done: true });
        return res.status(200).json({
            code: 200,
            message: "Welcome step marked as done",
            data: { is_welcome_done: true }
        });
    } catch (error) {
        console.error("Error in markWelcomeDone:", error);
        return res.status(500).json({
            code: 500,
            message: "Something went wrong",
            error: error.message
        });
    }
};

export const essentialsListing = async (req, res) => {
    try {
        const data = await userModel.essentialsListing();
        return res.status(200).json({
            code: 200,
            message: "Success",
            data
        });
    } catch (error) {
        console.error("Error in essentialsListing controller:", error);
        return res.status(500).json({
            code: 500,
            message: "Internal Server Error",
            error: error.message
        });
    }
};

export const addAgeGender = async (req, res) => {
    try {
        const userId = req.user_id;
        const { gender, age } = req.body;

        const updates = {};

        if ([0, 1].includes(gender)) {
            updates.gender = gender;
        }

        if ([0, 1].includes(age)) {
            updates.age = age;
            updates.on_boarding_count = 0;
        }

        if (Object.keys(updates).length === 0) {
            return res.status(400).json({
                code: 400,
                message: "No valid fields to update",
            });
        }

        await userModel.updateUserById(userId, updates);

        return res.status(200).json({
            code: 200,
            message: "Success",
            data: {
                on_boarding_count: updates.on_boarding_count ?? null,
            },
        });

    } catch (error) {
        console.error("Error in addAgeGender:", error);
        return res.status(500).json({
            code: 500,
            message: "Something went wrong",
        });
    }
};

export const getQuestionWithOptions = async (req, res) => {
    try {
        const questionId = req.query.question_id;
        const userId = req.user_id;

        const questionData = await userModel.getQuestionWithOptions(questionId, userId);

        if (!questionData) {
            return res.status(404).json({
                code: 404,
                message: "Question not found",
            });
        }

        if (questionId == 6) {
            const previousValueObj = await userModel.getLatestEssentialValue(userId, 5);
            const previousValue = previousValueObj?.value ?? null;
            const prevNum = previousValue ? parseInt(previousValue) : null;
            if (prevNum) {
                questionData.options = questionData.options.slice(0, prevNum);
            }
        }

        return res.status(200).json({
            code: 200,
            message: "Success",
            data: questionData,
        });

    } catch (error) {
        console.error("Error in getQuestionWithOptions controller:", error);
        return res.status(500).json({
            code: 500,
            message: "Internal Server Error",
        });
    }
};

export const getOptionsByEssentialsId = async (req, res) => {
    try {
        const essentialsId = parseInt(req.query.essentials_id);
        const userId = req.user_id;

        let options = await userModel.getOptionsByEssentialsId(essentialsId);

        if (essentialsId == 6) {
            const previousValueObj = await userModel.getLatestEssentialValue(userId, 5);
            const previousValue = previousValueObj?.value ?? null;
            const prevNum = previousValue ? parseInt(previousValue) : null;
            if (prevNum) {
                options = options.slice(0, prevNum);
            }
        }

        return res.status(200).json({
            code: 200,
            message: "Options fetched sucessfully",
            data: options
        });

    } catch (error) {
        console.error("Error in getOptionsByEssentialsId:", error);
        res.status(500).json({ message: "Internal server error" });
    }
};

export const saveUserEssential = async (req, res) => {
    const user_id = req.user_id;
    const gender = req.gender;  // 0 = Female, 1 = Male
    const age = req.age;       // 0 = Under 18, 1 = 18+
    const { date, day, essentials_id, value, on_boarding_count } = req.body;
    try {
        if (!essentials_id && on_boarding_count !== undefined) {
            await userModel.deleteEssentialsByCount(user_id, on_boarding_count);
            await userModel.updateUserById(user_id, { on_boarding_count });
            return res.status(200).json({
                code: 200,
                message: "Onboarding count updated successfully"
            });
        }

        const allowedEssentials = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11];
        if (!allowedEssentials.includes(Number(essentials_id))) {
            return res.status(400).json({
                code: 400,
                message: "Invalid essentials_id"
            });
        }

        let score = 0;
        if (Number(essentials_id) === 6) {
            const previousValueObj = await userModel.getLatestEssentialValue(user_id, 5);
            const previousValue = previousValueObj?.value ?? null;
            const prevNum = previousValue ? parseInt(previousValue) : null;
            const newNum = parseInt(value);
            if (prevNum !== null && newNum > prevNum) {
                return res.status(200).json({
                    code: 200,
                    message: `Value must be less than or equal to ${prevNum} meals`
                });
            }
            score = calculateEssentialScore({ essentials_id, value: previousValue, value2: value, gender, age });
            await userModel.updateEssentialScore(previousValueObj.id, user_id, score);
        } else {
            score = calculateEssentialScore({ essentials_id, value, gender, age });
        }

        await userModel.saveUserEssential(user_id, date, day, essentials_id, value, score);

        if (on_boarding_count !== undefined && on_boarding_count !== 0) {
            await userModel.updateUserById(user_id, { on_boarding_count });
        }

        return res.status(200).json({
            code: 200,
            message: "Essential record saved successfully"
        });

    } catch (error) {
        console.error("Error in saveUserEssential controller:", error);
        res.status(500).json({ code: 500, message: "Internal server error" });
    }
};

export const home = async (req, res) => {
    try {
        const user_id = req.user_id;
        const { date, day } = req.query;

        const startOfDay = Math.floor(new Date(new Date(date * 1000).setHours(0, 0, 0, 0)).getTime() / 1000);
        const endOfDay = Math.floor(new Date(new Date(date * 1000).setHours(23, 59, 59, 999)).getTime() / 1000);

        const [quote, user, essentials] = await Promise.all([
            userModel.getLatestQuote(startOfDay, endOfDay),
            userAuthModel.getUserById(user_id),
            userModel.getEssentials(user_id, startOfDay, endOfDay, day)
        ]);

        const totalScore = essentials.reduce((sum, item) => sum + (parseFloat(item.score) || 0), 0);

        return res.status(200).json({
            code: 200,
            message: "Success",
            user_name: user.user_name,
            profile_image: user.profile_image,
            totalScore: formatScore(totalScore),
            quote_text: quote?.quote_text || null,
            author_name: quote?.author_name || null,
            date,
            data: essentials
        });

    } catch (error) {
        console.error("Error in home controller:", error);
        return res.status(500).json({ code: 500, message: "Internal Server Error", error: error.message });
    }
};

export const editUserProfile = async (req, res) => {
    try {
        const userId = req.user_id;
        const { user_name, gender, age } = req.body;

        const updates = {};
        if (user_name?.trim()) updates.user_name = user_name.trim();
        if ([0, 1].includes(+gender)) updates.gender = +gender;
        if ([0, 1].includes(+age)) updates.age = +age;
        if (req.file?.filename) updates.profile_image = req.file.filename;

        if (!Object.keys(updates).length)
            return res.status(400).json({ code: 400, message: "No valid fields to update" });

        await userModel.updateUserById(userId, updates);
        const user = await userAuthModel.getUserById(userId);

        return res.status(200).json({
            code: 200,
            message: "Profile updated successfully",
            data: user
        });

    } catch (error) {
        console.error("Error in editUserProfile:", error);
        return res.status(500).json({ code: 500, message: "Something went wrong" });
    }
};

export const getNotifications = async (req, res) => {
    try {
        const user_id = req.user_id;
        let { page = 1, limit = 10 } = req.query;
        page = parseInt(page, 10);
        limit = parseInt(limit, 10);

        const { data, total } = await userModel.getNotifications(user_id, page, limit);

        return res.status(200).json({
            code: 200,
            message: "Notifications fetched successfully",
            data,
            pagination: {
                totalRecords: total,
                totalPages: Math.ceil(total / limit),
                currentPage: page,
                limit
            }
        });

    } catch (error) {
        console.error("Error in getNotifications:", error);
        return res.status(500).json({
            code: 500,
            message: "Internal Server Error",
            error: error.message
        });
    }
};

export const deleteNotification = async (req, res) => {
    try {
        const user_id = req.user_id;
        const { id } = req.query;

        await userModel.deleteNotification(user_id, id);

        return res.status(200).json({
            code: 200,
            message: "Notification deleted successfully",
        });

    } catch (error) {
        console.error("❌ Error in deleteNotification:", error);
        return res.status(500).json({
            code: 500,
            message: "Internal Server Error",
            error: error.message,
        });
    }
};

export const essentialsTips = async (req, res) => {
    try {
        const { essentials_id, page = 1, limit = 10 } = req.query;
        const currentPage = parseInt(page, 10);
        const pageLimit = parseInt(limit, 10);

        const { data: tips, total } = await userModel.essentialsTips(essentials_id, currentPage, pageLimit);

        const formattedTips = tips.map((tip, index) => ({
            id: tip.id,
            display_id: (currentPage - 1) * pageLimit + index + 1,
            essentials_id: tip.essentials_id,
            essentials_tips: tip.essentials_tips
        }));

        return res.status(200).json({
            code: 200,
            message: "List fetched successfully",
            data: formattedTips,
            pagination: {
                totalRecords: total,
                totalPages: Math.ceil(total / pageLimit),
                currentPage,
                limit: pageLimit
            }
        });

    } catch (error) {
        console.error("Error in essentialsTips controller:", error);
        res.status(500).json({ code: 500, message: "Internal server error" });
    }
};

export const getUserAvailableMonths = async (req, res) => {
    try {
        const user_id = req.user_id;

        const months = await userModel.fetchUserMonths(user_id);

        return res.status(200).json({
            code: 200,
            message: "Available months fetched successfully",
            data: months
        });

    } catch (error) {
        console.error("Error in getUserAvailableMonths:", error);
        return res.status(500).json({
            code: 500,
            message: "Internal server error"
        });
    }
};

// export const getUserProgress = async (req, res) => {
//     try {
//         const user_id = req.user_id;
//         const data = await fetchProgress(user_id);
//         return res.status(200).json({
//             code: 200,
//             message: "User progress fetched successfully",
//             data
//         });
//     } catch (error) {
//         console.error("Error in getUserProgress:", error);
//         return res.status(500).json({ code: 500, message: "Internal server error" });
//     }
// };

export const getUserProgress = async (req, res) => {
    try {
        const user_id = req.user_id;
        const month_epoch = req.query.month_epoch ? parseInt(req.query.month_epoch) : null;

        const data = await fetchProgress(user_id, month_epoch);

        return res.status(200).json({
            code: 200,
            message: "User progress fetched successfully",
            data
        });

    } catch (error) {
        console.error("Error in getUserProgress:", error);
        return res.status(500).json({ code: 500, message: "Internal server error" });
    }
};

export const getSuggestions = async (req, res) => {
    try {
        const user_id = req.user_id;
        const { focusAreas } = await fetchProgress(user_id);
        if (!focusAreas.length) {
            return res.status(200).json({
                code: 200,
                message: "No focus areas found",
                data: []
            });
        }
        return res.status(200).json({
            code: 200,
            message: "Suggested focus areas fetched successfully",
            data: focusAreas
        });
    } catch (error) {
        console.error("Error in getSuggestions controller:", error);
        return res.status(500).json({ code: 500, message: "Internal server error" });
    }
};