import { errorMessage } from "../../../../utils/errorMessage.js";
import { successMessage } from "../../../../utils/successMessage.js";
import * as subscriptionModel from "../../../../models/subscriptionModel.js";
import logger from "../../../../utils/logger.js";
import { oauth2Client, SCOPE } from "../../../../services/subscription/subscriptionService.js";


export const getGoogleAuthUrl = async (req, res) => {
    try {
        const authUrl = oauth2Client.generateAuthUrl({
            access_type: "offline",
            scope: [SCOPE],
            prompt: "consent",
        });

        return res.status(200).json({
            code: 200,
            message: "Google authorization URL generated",
            data: { authUrl },
        });

    } catch (error) {

        logger.error({
            route: req.originalUrl,
            method: req.method,
            message: "Error generating Google auth URL",
            error: error.message,
        });

        return res.status(500).json({
            code: 500,
            message: "Something went wrong",
        });
    }
};

export const googleCallback = async (req, res) => {
    try {
        const { code } = req.query;
        if (!code) {
            return res.status(400).json({
                code: 400,
                message: "Authorization code not provided",
            });
        }

        const { tokens } = await oauth2Client.getToken(code);
        const accessTokenExpiresAt = Math.floor(tokens.expiry_date / 1000);
        const refreshTokenExpiresAt = tokens.refresh_token_expires_in
            ? Math.floor(Date.now() / 1000) + tokens.refresh_token_expires_in
            : null;

        await subscriptionModel.storeGoogleTokens(
            tokens.access_token,
            tokens.refresh_token,
            accessTokenExpiresAt,
            refreshTokenExpiresAt
        );

        logger.info({
            route: req.originalUrl,
            method: req.method,
            message: "Google tokens stored",
            data: {
                access_token: tokens.access_token,
                refresh_token: tokens.refresh_token,
                access_token_expires_at: accessTokenExpiresAt,
                refresh_token_expires_at: refreshTokenExpiresAt,
            },
        });

        return res.status(200).json({
            code: 200,
            message: "Tokens stored successfully",
            data: {
                access_token: tokens.access_token,
                refresh_token: tokens.refresh_token,
                access_token_expires_in: accessTokenExpiresAt - Math.floor(Date.now() / 1000),
                refresh_token_expires_in: refreshTokenExpiresAt
                    ? refreshTokenExpiresAt - Math.floor(Date.now() / 1000)
                    : null,
            },
        });

    } catch (error) {
        logger.error({
            route: req.originalUrl,
            method: req.method,
            message: "Error handling Google callback",
            error: error.message,
        });

        return res.status(500).json({
            code: 500,
            message: errorMessage.SOMETHING_WRONG.message,
        });
    }
};

export const addSubscription = async (req, res) => {
    try {
        const user_id = req.user_id;
        const {
            plan_id,
            subscription_id,
            expires_at,
            plan_price,
            product_id,
            device_type,
            package_name,
            order_id,
            purchase_token,
            transaction_id,
            plan_duration,
        } = req.body;

        if (!plan_id || !subscription_id || !expires_at || !plan_price || !product_id || !device_type) {
            return res.status(errorMessage.MISSING_REQUIRED_FIELDS.code).json({
                code: errorMessage.MISSING_REQUIRED_FIELDS.code,
                message: errorMessage.MISSING_REQUIRED_FIELDS.message,
            });
        }

        await subscriptionModel.insertSubscription({
            user_id,
            plan_id,
            subscription_id,
            expires_at,
            plan_price,
            product_id,
            device_type,
            package_name,
            order_id,
            purchase_token,
            transaction_id,
            plan_duration,
            status: "active",
            event_type: "new",
        });

        return res.status(successMessage.SUBSCRIPTION_ADDED.code).json({
            code: successMessage.SUBSCRIPTION_ADDED.code,
            message: successMessage.SUBSCRIPTION_ADDED.message
        });

    } catch (error) {
        logger.error({
            route: req.originalUrl,
            method: req.method,
            message: "Error adding subscription",
            error: error.message,
        });

        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
        });
    }
};

export const cancelSubscription = async (req, res) => {
    const user_id = req.user_id;
    try {
        const subscription = await subscriptionModel.getActiveSubscription(user_id);

        if (!subscription) {
            return res.status(errorMessage.NO_SUBSCRIPTION_FOUND.code).json({
                code: errorMessage.NO_SUBSCRIPTION_FOUND.code,
                message: errorMessage.NO_SUBSCRIPTION_FOUND.message,
            });
        }

        if (subscription.device_type === "Android") {
            return res.status(400).json({
                code: 400,
                message: "Android subscriptions cannot be cancelled manually.",
            });
        }

        // here cancel old all subscriptions then insert new one

        await subscriptionModel.updateSubscriptionStatus(
            user_id,
            subscription.id,
            "cancelled",
            "cancelled"
        );

        return res.status(successMessage.SUBSCRIPTION_CANCELLED.code).json({
            code: successMessage.SUBSCRIPTION_CANCELLED.code,
            message: successMessage.SUBSCRIPTION_CANCELLED.message,
        });

    } catch (error) {

        console.error(`Error cancelling subscription for user ${user_id}:`, error.message);
        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
        });

    }
};

export const checkSubscription = async (req, res) => {
    try {
        const user_id = req.user_id;
        const subscription = await subscriptionModel.getActiveSubscription(user_id);

        if (!subscription) {
            return res.status(successMessage.SUBSCRIPTION_NOT_FOUND.code).json({
                code: successMessage.SUBSCRIPTION_NOT_FOUND.code,
                message: successMessage.SUBSCRIPTION_NOT_FOUND.message,
            });
        }

        // const now = Math.floor(Date.now() / 1000);
        // if (subscription.device_type === "Apple" && now > subscription.expires_at) {
        //     await subscriptionModel.updateSubscriptionStatus(user_id, subscription.id, "expired", "expired");
        //     return res.status(successMessage.SUBSCRIPTION_EXPIRED.code).json({
        //         code: successMessage.SUBSCRIPTION_EXPIRED.code,
        //         message: successMessage.SUBSCRIPTION_EXPIRED.message
        //     });
        // }

        return res.status(successMessage.SUBSCRIPTION_ACTIVE.code).json({
            code: successMessage.SUBSCRIPTION_ACTIVE.code,
            message: successMessage.SUBSCRIPTION_ACTIVE.message,
            data: subscription,
        });

    } catch (error) {
        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
        });
    }
};