import { generateUserToken } from "../../../../middlewares/userAuthMiddleware.js";
import { errorMessage } from "../../../../utils/errorMessage.js";
import { successMessage } from "../../../../utils/successMessage.js";
import { sendEmail, loadHtmlTemplate } from "../../../../services/email/emailService.js";
import { hashPassword, compPassword, generateRandomPassword } from "../../../../utils/passwordHasher.js";
import * as userAuthModel from "../../../../models/userAuthModel.js";
import { generateOtp } from "../../../../utils/commonFunctions.js";
// import * as subscriptionModel from "../../../../models/subscriptionModel.js";
// const subscription = await subscriptionModel.getActiveSubscription(userData.id);
const BASE_URL = process.env.BASE_URL;

// ----------------------------- Public functions ----------------------------------

export const signUp = async (req, res) => {
    try {
        const { user_name, email, password, confirm_password } = req.body;

        if (password !== confirm_password) {
            return res.status(errorMessage.USER_PASS_CNF_PASS.code).json({
                code: errorMessage.USER_PASS_CNF_PASS.code,
                message: errorMessage.USER_PASS_CNF_PASS.message,
            });
        }

        const emailExist = await userAuthModel.checkUserEmail(email);
        if (emailExist) {
            return res.status(errorMessage.USER_EMAIL_ALREADY_EXIST.code).json({
                code: errorMessage.USER_EMAIL_ALREADY_EXIST.code,
                message: errorMessage.USER_EMAIL_ALREADY_EXIST.message,
            });
        }

        const encryptedPassword = await hashPassword(password);
        const userData = { user_name, email, password: encryptedPassword };
        const result = await userAuthModel.signUp(userData);
        const insertedUserId = result.insertId;

        await sendEmail({
            to: email,
            subject: "Primal5 Email Verification",
            templateName: "verificationEmail",
            variables: {
                USER_NAME: user_name,
                VERIFY_URL: `${BASE_URL}/api/v1/user/auth/verify_email/${insertedUserId}`,
                BASE_URL,
            },
        });

        const user_details = await userAuthModel.getUserById(insertedUserId);
        if (user_details) delete user_details.password;

        return res.status(successMessage.USER_REGISTER.code)
            .json({
                code: successMessage.USER_REGISTER.code,
                message: successMessage.USER_REGISTER.message,
                data: user_details,
            });

    } catch (error) {
        console.error("Error in signUp:", error);
        return res.status(errorMessage.SOMETHING_WRONG.code)
            .json({
                code: errorMessage.SOMETHING_WRONG.code,
                message: errorMessage.SOMETHING_WRONG.message,
            });
    }
};

export const login = async (req, res) => {
    try {
        const { email, password, device_token, device_type, time_zone } = req.body;

        const userData = await userAuthModel.checkUserEmail(email);
        if (!userData) {
            return res.status(errorMessage.USER_NOT_EXIST.code).json({
                code: errorMessage.USER_NOT_EXIST.code,
                message: errorMessage.USER_NOT_EXIST.message,
            });
        }

        if (userData.is_blocked_by_admin === 1) {
            return res.status(errorMessage.IS_BLOCKED_BY_ADMIN.code).json({
                code: errorMessage.IS_BLOCKED_BY_ADMIN.code,
                message: errorMessage.IS_BLOCKED_BY_ADMIN.message,
            });
        }

        const isPasswordMatch = await compPassword(password, userData.password);
        if (!isPasswordMatch) {
            return res.status(errorMessage.INCORRECT_PASSWORD.code).json({
                code: errorMessage.INCORRECT_PASSWORD.code,
                message: errorMessage.INCORRECT_PASSWORD.message,
            });
        }

        if (userData.is_email_verified === 0) {
            return res.status(400).json({
                code: errorMessage.EMAIL_NOT_VERIFIED_YET.code,
                message: errorMessage.EMAIL_NOT_VERIFIED_YET.message,
            });
        }

        const access_token = generateUserToken({ user_id: userData.id });
        await userAuthModel.updateUsersTable(access_token, { device_token, device_type, time_zone }, userData.id);
        const userDetails = await userAuthModel.getUserById(userData.id);

        if (userDetails) {
            delete userDetails.password;
            userDetails.access_token = access_token;
            userDetails.device_token = device_token || userDetails.device_token;
            userDetails.device_type = device_type || userDetails.device_type;
        }

        return res.status(successMessage.USER_LOGGED_IN.code).json({
            code: successMessage.USER_LOGGED_IN.code,
            message: successMessage.USER_LOGGED_IN.message,
            data: userDetails,
        });

    } catch (error) {
        console.error("Error during login:", error);
        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
        });
    }
};

export const forgotPassword = async (req, res) => {
    try {
        const { email } = req.body;

        const userData = await userAuthModel.checkUserEmail(email);
        if (!userData) {
            return res.status(errorMessage.PASSWORD_RESET_MAIL_NOT_EXIST.code).json({
                code: errorMessage.PASSWORD_RESET_MAIL_NOT_EXIST.code,
                message: errorMessage.PASSWORD_RESET_MAIL_NOT_EXIST.message,
            });
        }

        if (userData.is_blocked_by_admin === 1) {
            return res.status(errorMessage.IS_BLOCKED_BY_ADMIN.code).json({
                code: errorMessage.IS_BLOCKED_BY_ADMIN.code,
                message: errorMessage.IS_BLOCKED_BY_ADMIN.message,
            });
        }

        const newPassword = generateRandomPassword();
        const hashedPassword = await hashPassword(newPassword);
        await userAuthModel.updatePassword(userData.id, hashedPassword);

        await sendEmail({
            to: email,
            subject: "Primal5 - Password Reset",
            templateName: "resetPasswordEmail",
            variables: {
                USER_NAME: userData.user_name,
                PASSWORD: newPassword,
                BASE_URL,
            },
        });

        return res.status(successMessage.PASSWORD_RESET_MAIL.code).json({
            code: successMessage.PASSWORD_RESET_MAIL.code,
            message: successMessage.PASSWORD_RESET_MAIL.message,
        });

    } catch (error) {
        console.error("Error in forgotPassword:", error);
        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
        });
    }
};

export const googleLogin = async (req, res) => {
    try {
        const { email, device_token, device_type, google_token, user_name, time_zone } = req.body;

        const userData = await userAuthModel.checkUserEmail(email);
        let access_token;
        let user_details;

        if (userData) {

            if (userData.is_blocked_by_admin === 1) {
                return res.status(errorMessage.IS_BLOCKED_BY_ADMIN.code).json({
                    code: errorMessage.IS_BLOCKED_BY_ADMIN.code,
                    message: errorMessage.IS_BLOCKED_BY_ADMIN.message,
                });
            }
            access_token = generateUserToken({ user_id: userData.id });
            await userAuthModel.updateUsersTableInGoogleLogin(access_token, req.body, userData.id);
            user_details = await userAuthModel.getUserById(userData.id);

        } else {

            const result = await userAuthModel.signUp(req.body);
            const insertedUserId = result.insertId;
            access_token = generateUserToken({ user_id: insertedUserId });
            await userAuthModel.updateUsersTable(access_token, req.body, insertedUserId);
            user_details = await userAuthModel.getUserById(insertedUserId);

        }

        delete user_details.password;
        user_details.google_token = google_token || user_details.google_token;
        user_details.access_token = access_token || user_details.access_token;
        user_details.device_token = device_token || user_details.device_token;
        user_details.device_type = device_type || user_details.device_type;

        return res.status(successMessage.USER_LOGGED_IN.code).json({
            code: successMessage.USER_LOGGED_IN.code,
            message: successMessage.USER_LOGGED_IN.message,
            data: user_details,
        });

    } catch (error) {
        console.error("Error in googleLogin:", error);
        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
        });
    }
};

export const appleLogin = async (req, res) => {
    try {
        const { email, user_name, device_token, device_type, apple_token, profile_image, time_zone } = req.body;

        let user = null;
        let access_token;
        let user_details;

        user = await userAuthModel.getUserByAppleToken(apple_token);

        if (!user && email) {
            user = await userAuthModel.checkUserEmail(email);
            if (user) {
                await userAuthModel.updateAppleToken(user.id, apple_token);
            }
        }

        if (user) {

            if (user.is_blocked_by_admin === 1) {
                return res.status(errorMessage.IS_BLOCKED_BY_ADMIN.code).json({
                    code: errorMessage.IS_BLOCKED_BY_ADMIN.code,
                    message: errorMessage.IS_BLOCKED_BY_ADMIN.message,
                });
            }
            access_token = generateUserToken({ user_id: user.id });
            await userAuthModel.updateAppleUser(user.id, { access_token, device_token, device_type, time_zone });
            user_details = await userAuthModel.getUserById(user.id);

        } else {

            const deletedUser = await userAuthModel.getDeletedAppleUser(apple_token);
            let result;
            if (deletedUser) {
                result = await userAuthModel.createAppleUser({
                    email: deletedUser.email,
                    user_name,
                    apple_token,
                    device_token,
                    device_type,
                    profile_image: deletedUser.profile_image,
                });
            } else {
                result = await userAuthModel.createAppleUser({
                    email,
                    user_name,
                    apple_token,
                    device_token,
                    device_type,
                    profile_image
                });
            }

            const insertedUserId = result.insertId;
            access_token = generateUserToken({ user_id: insertedUserId });
            await userAuthModel.updateAppleUser(insertedUserId, { access_token, device_token, device_type, time_zone });
            user_details = await userAuthModel.getUserById(insertedUserId);

        }

        delete user_details.password;
        user_details.apple_token = apple_token || user_details.apple_token;
        user_details.access_token = access_token;
        user_details.device_token = device_token || user_details.device_token;
        user_details.device_type = device_type || user_details.device_type;

        return res.status(successMessage.USER_LOGGED_IN.code).json({
            code: successMessage.USER_LOGGED_IN.code,
            message: successMessage.USER_LOGGED_IN.message,
            data: user_details,
        });

    } catch (error) {
        console.error("Error in appleLogin:", error);
        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
        });
    }
};

export const verifyEmail = async (req, res) => {
    try {
        const userId = req.params.id;
        const user = await userAuthModel.getUserById(userId);

        if (!user) {
            const html = await loadHtmlTemplate("verificationFailure");
            return res.status(404).type("html").send(html);
        }

        if (user.is_email_verified === 1) {
            const html = await loadHtmlTemplate("alreadyVerified");
            return res.status(200).type("html").send(html);
        }

        const success = await userAuthModel.verifyEmail(userId);
        if (!success) {
            const html = await loadHtmlTemplate("verificationFailure");
            return res.status(400).type("html").send(html);
        }

        const html = await loadHtmlTemplate("verified");
        return res.status(200).type("html").send(html);

    } catch (error) {
        console.error("❌ Verification error:", error.message);
        const html = await loadHtmlTemplate("verificationFailure");
        return res.status(500).type("html").send(html);
    }
};

export const sendOtpForAccountDeletion = async (req, res) => {
    try {
        const { email } = req.body;

        const user = await userAuthModel.checkUserEmail(email);
        if (!user) {
            return res.render("users/deleteAccountOtp", { message: "Email not found." });
        }

        const otp = generateOtp();
        await userAuthModel.saveAccountDeletionOtp(user.id, otp);

        await sendEmail({
            to: email,
            subject: "Primal5 Account Deletion OTP",
            templateName: "accountDeletionOtp",
            variables: {
                USER_NAME: user.user_name,
                OTP: otp,
                BASE_URL
            },
        });

        res.render("users/deleteAccountVerify", { email, message: "OTP sent successfully" });

    } catch (error) {
        console.error("Error in sendOtpForAccountDeletion:", error);
        res.render("users/deleteAccountOtp", { message: "Something went wrong. Try again." });
    }
};

export const verifyOtpAndDeleteAccount = async (req, res) => {
    try {
        let { email, otp } = req.body;

        const user = await userAuthModel.checkUserEmail(email);
        if (!user) {
            return res.render('users/deleteAccountVerify', { email, message: "Email not found" });
        }

        if (user.account_delete_otp !== otp) {
            return res.render('users/deleteAccountVerify', { email, message: "Wrong OTP" });
        }

        const result = await userAuthModel.deleteAccount(user.id);

        const message = result.affectedRows > 0
            ? "Account deleted successfully"
            : "Failed to delete account. Try again.";

        return res.render('users/deleteAccountVerify', { email: result.affectedRows > 0 ? "" : email, message });

    } catch (err) {
        console.error("Error in verifyOtpAndDeleteAccount:", err);
        return res.status(500).json({ code: 500, message: "Internal Server Error" });
    }
};

// ---------------------------- Protected functions ---------------------------------

export const helpAndSupport = async (req, res) => {
    try {
        const user_id = req.user_id;
        const user_name = req.user_name;
        const { email, message } = req.body;

        await userAuthModel.saveHelpAndSupport({ user_id, user_name, email, message });

        await sendEmail({
            to: email,
            subject: `Help & Support Request by - ${user_name}`,
            templateName: "helpAndSupport",
            variables: {
                USER_NAME: user_name,
                EMAIL: email,
                MESSAGE: message,
                BASE_URL,
            },
        });

        return res.status(successMessage.MAIL_SEND_SMTP.code).json({
            code: successMessage.MAIL_SEND_SMTP.code,
            message: successMessage.MAIL_SEND_SMTP.message,
        });

    } catch (error) {
        console.error("Help & Support Error:", error);
        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
        });
    }
};

export const changePassword = async (req, res) => {
    try {
        const { old_password, new_password, confirm_password } = req.body;
        const user_id = req.user_id;

        if (new_password === old_password) {
            return res.status(errorMessage.NEW_OLD_PASSWORD_SAME.code).json({
                code: errorMessage.NEW_OLD_PASSWORD_SAME.code,
                message: errorMessage.NEW_OLD_PASSWORD_SAME.message,
            });
        }

        if (new_password !== confirm_password) {
            return res.status(errorMessage.PASSWORD_UNMATCH.code).json({
                code: errorMessage.PASSWORD_UNMATCH.code,
                message: errorMessage.PASSWORD_UNMATCH.message,
            });
        }

        const user = await userAuthModel.checkUserById(user_id);
        if (!user) {
            return res.status(errorMessage.USER_NOT_EXIST.code).json({
                code: errorMessage.USER_NOT_EXIST.code,
                message: errorMessage.USER_NOT_EXIST.message,
            });
        }

        const isPasswordMatch = await compPassword(old_password, user.password);
        if (!isPasswordMatch) {
            return res.status(errorMessage.INVALID_OLD_PASSWORD.code).json({
                code: errorMessage.INVALID_OLD_PASSWORD.code,
                message: errorMessage.INVALID_OLD_PASSWORD.message,
            });
        }

        const hashedPassword = await hashPassword(new_password);
        await userAuthModel.updatePassword(req.user_id, hashedPassword);

        return res.status(successMessage.PASSWORD_CHANGED_SUCCESS.code).json({
            code: successMessage.PASSWORD_CHANGED_SUCCESS.code,
            message: successMessage.PASSWORD_CHANGED_SUCCESS.message,
        });

    } catch (error) {
        console.error("Error during changePassword:", error.message);
        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
        });
    }
};

export const logout = async (req, res) => {
    try {
        const userId = req.user_id;
        await userAuthModel.logout(userId);
        return res.status(successMessage.USER_LOGGED_OUT.code).json({
            code: successMessage.USER_LOGGED_OUT.code,
            message: successMessage.USER_LOGGED_OUT.message,
        });
    } catch (error) {
        console.error("Logout Error:", error);
        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
        });
    }
};

export const deleteAccount = async (req, res) => {
    try {
        const userId = req.user_id;
        const result = await userAuthModel.deleteAccount(userId);
        return res
            .status(successMessage.DELETED.code)
            .json({
                code: successMessage.DELETED.code,
                message: successMessage.DELETED.message
            });
    } catch (error) {
        console.error("❌ Error in deleteAccount:", error);
        return res
            .status(errorMessage.SOMETHING_WRONG.code)
            .json({
                code: errorMessage.SOMETHING_WRONG.code,
                message: errorMessage.SOMETHING_WRONG.message
            });
    }
};