import { generateAdminToken } from "../../../../middlewares/adminAuthMiddleware.js";
import { errorMessage } from "../../../../utils/errorMessage.js";
import { successMessage } from "../../../../utils/successMessage.js";
import { hashPassword, compPassword } from "../../../../utils/passwordHasher.js";
import * as adminAuthModel from "../../../../models/adminAuthModel.js";
import { formatUrl } from "../../../../utils/commonFunctions.js";


export const login = async (req, res) => {
    const { email, password } = req.body;
    console.log("-----------logout api hitted-----------");
    try {
        const admin = await adminAuthModel.getAdminByEmail(email);
        if (!admin) {
            return res.status(errorMessage.USER_NOT_EXIST.code).json({
                code: errorMessage.USER_NOT_EXIST.code,
                message: errorMessage.USER_NOT_EXIST.message,
            });
        }

        const isMatch = await compPassword(password, admin.password);
        if (!isMatch) {
            return res.status(errorMessage.INCORRECT_PASSWORD.code).json({
                code: errorMessage.INCORRECT_PASSWORD.code,
                message: errorMessage.INCORRECT_PASSWORD.message,
            });
        }

        const access_token = await generateAdminToken({ user_id: admin.id });
        await adminAuthModel.updateAdminToken(admin.id, access_token);

        res.cookie("access_token", access_token, {
            httpOnly: true,
            secure: process.env.COOKIE_SECURE === "true",
            sameSite: "Strict",
            maxAge: 60 * 60 * 1000
        });

        return res.status(successMessage.USER_LOGGED_IN.code).json({
            code: successMessage.USER_LOGGED_IN.code,
            message: successMessage.USER_LOGGED_IN.message
        });

    } catch (error) {
        console.error("❌ Admin Login Error:", error.message);
        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
            error: error.message || error
        });
    }
};

export const logout = async (req, res) => {
    const user_id = req.user_id;
    try {
        await adminAuthModel.logoutAdmin(user_id);

        res.clearCookie("access_token", {
            httpOnly: true,
            secure: process.env.COOKIE_SECURE === "true",
            sameSite: "Strict",
        });

        return res.status(successMessage.USER_LOGGED_OUT.code).json({
            code: successMessage.USER_LOGGED_OUT.code,
            message: successMessage.USER_LOGGED_OUT.message
        });

    } catch (err) {
        res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message
        });
    }
};

export const changePassword = async (req, res) => {
    try {
        const { old_password, new_password, confirm_password } = req.body;
        const user_id = req.user_id;

        if (new_password === old_password) {
            return res.status(errorMessage.NEW_OLD_PASSWORD_SAME.code).json({
                code: errorMessage.NEW_OLD_PASSWORD_SAME.code,
                message: errorMessage.NEW_OLD_PASSWORD_SAME.message,
            });
        }

        if (new_password !== confirm_password) {
            return res.status(errorMessage.PASSWORD_UNMATCH.code).json({
                code: errorMessage.PASSWORD_UNMATCH.code,
                message: errorMessage.PASSWORD_UNMATCH.message,
            });
        }

        const admin = await adminAuthModel.checkAdminById(user_id);
        if (!admin) {
            return res.status(errorMessage.USER_NOT_EXIST.code).json({
                code: errorMessage.USER_NOT_EXIST.code,
                message: errorMessage.USER_NOT_EXIST.message,
            });
        }

        const isPasswordMatch = await compPassword(old_password, admin.password);
        if (!isPasswordMatch) {
            return res.status(errorMessage.INVALID_OLD_PASSWORD.code).json({
                code: errorMessage.INVALID_OLD_PASSWORD.code,
                message: errorMessage.INVALID_OLD_PASSWORD.message,
            });
        }

        const hashedPassword = await hashPassword(new_password);
        await adminAuthModel.updatePassword(req.user_id, hashedPassword);

        return res.status(successMessage.PASSWORD_CHANGED_SUCCESS.code).json({
            code: successMessage.PASSWORD_CHANGED_SUCCESS.code,
            message: successMessage.PASSWORD_CHANGED_SUCCESS.message,
        });

    } catch (error) {
        console.error("Error during changePassword:", error.message);
        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
        });
    }
};

export const getAdminProfile = async (req, res) => {
    try {
        const user_id = req.user_id;
        const admin = await adminAuthModel.checkAdminById(user_id);

        return res.status(successMessage.LOGGED_IN_USER_PROFILE.code).json({
            code: successMessage.LOGGED_IN_USER_PROFILE.code,
            message: successMessage.LOGGED_IN_USER_PROFILE.message,
            data: {
                id: admin.id,
                user_name: admin.user_name,
                profile_image: formatUrl(admin.profile_image),
            },
        });

    } catch (error) {
        console.error("Error in getUserProfile:", error);
        return res.status(errorMessage.SOMETHING_WRONG.code).json({
            code: errorMessage.SOMETHING_WRONG.code,
            message: errorMessage.SOMETHING_WRONG.message,
        });
    }
};