import XLSX from "xlsx";
import { errorMessage } from "../../../../utils/errorMessage.js";
import { successMessage } from "../../../../utils/successMessage.js";
import { sendAndroidPush, sendApplePush } from "../../../../services/notifications/pushNotificationService.js";
import * as adminModel from "../../../../models/adminModel.js";
import * as userModel from "../../../../models/userModel.js";
import logger from "../../../../utils/logger.js";
import { fetchProgress } from '../../../../utils/commonFunctions.js';


export const massPush = async (req, res) => {
  try {
    const { title = "", message = "" } = req.body;

    const users = await adminModel.getAllUsersWithDeviceTokens();
    if (!users.length) {
      logger.warn("No users found for mass push");
      return res.status(404).json({ code: 404, message: "No users found." });
    }

    const androidTokens = [];
    const iosTokens = [];
    const notificationRecords = [];
    for (const user of users) {
      if (!user.device_token) continue;
      if (user.device_type === "Android") androidTokens.push(user.device_token);
      else if (user.device_type === "Apple") iosTokens.push(user.device_token);
      notificationRecords.push({
        user_id: user.id,
        title,
        message,
        pushType: 0,
      });
    }

    if (!androidTokens.length && !iosTokens.length) {
      logger.warn("No valid device tokens found");
      return res.status(400).json({ code: 400, message: "No valid device tokens." });
    }

    const notificationPayload = { title, message };

    const [androidResult, iosResult] = await Promise.allSettled([
      androidTokens.length ? sendAndroidPush(notificationPayload, androidTokens) : Promise.resolve({ total: 0, success: 0, failure: 0, failedItems: [] }),
      iosTokens.length ? sendApplePush(notificationPayload, iosTokens) : Promise.resolve({ total: 0, success: 0, failure: 0, failedItems: [] }),
    ]);

    await userModel.insertNotifications(notificationRecords);

    const androidSuccess = androidResult?.value?.success ?? 0;
    const iosSuccess = iosResult?.value?.success ?? 0;
    const androidFailure = androidResult?.value?.failure ?? androidTokens.length;
    const iosFailure = iosResult?.value?.failure ?? iosTokens.length;

    const totalSuccess = androidSuccess + iosSuccess;
    const totalFailure = androidFailure + iosFailure;

    const failedItems = [
      ...(androidResult?.value?.failedItems || []),
      ...(iosResult?.value?.failedItems || []),
    ];

    logger.info(`Mass push summary: ${totalSuccess} succeeded, ${totalFailure} failed`);
    if (failedItems.length) logger.error(`Mass push failures: ${failedItems.length} devices failed`, failedItems);

    await adminModel.insertMassPushHistory({
      title,
      message,
      totalUsers: users.length,
      successCount: totalSuccess,
      failureCount: totalFailure,
      androidSent: androidSuccess,
      iosSent: iosSuccess,
    });

    return res.status(200).json({
      code: 200,
      message: "Push notifications sent successfully.",
      summary: {
        totalUsers: users.length,
        successCount: totalSuccess,
        failureCount: totalFailure,
        androidSent: androidSuccess,
        iosSent: iosSuccess,
        failedItems,
      },
    });

  } catch (error) {
    logger.error("Error in massPush:", error);
    return res.status(500).json({
      code: errorMessage.SOMETHING_WRONG.code,
      message: errorMessage.SOMETHING_WRONG.message,
    });
  }
};

export const getMassPushHistory = async (req, res) => {
  try {
    const { search = "", page = 1, limit = 10 } = req.query;
    const pageNum = Math.max(parseInt(page, 10) || 1, 1);
    const limitNum = Math.min(Math.max(parseInt(limit, 10) || 10, 1), 100);

    const { total, rows } = await adminModel.getMassPushHistory({
      search,
      page: pageNum,
      limit: limitNum,
    });

    const numberedRows = rows.map((row, index) => ({
      sl_no: (pageNum - 1) * limitNum + index + 1,
      ...row,
    }));

    return res.status(200).json({
      code: 200,
      message: rows.length ? "Mass push history fetched successfully." : "No mass push history found.",
      data: rows,
      data: numberedRows,
      pagination: {
        totalRecords: total,
        totalPages: Math.ceil(total / limitNum),
        currentPage: pageNum,
        limit: limitNum,
      },
    });

  } catch (error) {
    logger.error("Error in getMassPushHistory:", error);
    return res.status(500).json({
      code: 500,
      message: "Something went wrong while fetching mass push history.",
    });
  }
};

export const getUsersList = async (req, res) => {
  try {
    let { page = 1, limit = 10, search = "" } = req.query;
    page = parseInt(page, 10);
    limit = parseInt(limit, 10);

    const { data, total } = await adminModel.getUsersList(page, limit, search);

    return res.status(200).json({
      code: 200,
      message: "Users fetched successfully",
      data,
      pagination: {
        totalRecords: total,
        totalPages: Math.ceil(total / limit),
        currentPage: page,
        limit
      }
    });

  } catch (error) {
    console.error("Error in getUsersList:", error);
    return res.status(500).json({
      code: 500,
      message: "Internal Server Error",
      error: error.message
    });
  }
};

export const updateUserStatus = async (req, res) => {
  try {
    const { id, blocked_by_admin } = req.body;

    const newStatus = blocked_by_admin == 1 ? 1 : 0;
    await adminModel.updateUserStatus(id, newStatus);

    return res.status(200).json({
      code: 200,
      message: newStatus
        ? "User blocked successfully."
        : "User unblocked successfully.",
      data: { id, is_blocked_by_admin: newStatus }
    });

  } catch (error) {
    console.error("Error in updateUserStatus:", error);
    return res.status(500).json({
      code: 500,
      message: "Internal Server Error",
      error: error.message
    });
  }
};

// export const getSubscribedUsersList = async (req, res) => {
//   try {
//     let { page = 1, limit = 10, search = "" } = req.query;
//     page = parseInt(page, 10);
//     limit = parseInt(limit, 10);

//     const { data, total } = await adminModel.getSubscribedUsersList(page, limit, search);
//     return res.status(200).json({
//       code: 200,
//       message: "Subscribed users fetched successfully",
//       data,
//       pagination: {
//         totalRecords: total,
//         totalPages: Math.ceil(total / limit),
//         currentPage: page,
//         limit
//       }
//     });

//   } catch (error) {
//     console.error("Error in getSubscribedUsersList:", error);
//     return res.status(500).json({
//       code: 500,
//       message: "Internal Server Error",
//       error: error.message
//     });
//   }
// };

export const userDetails = async (req, res) => {
  try {
    const userId = parseInt(req.params.id, 10);
    if (!userId) {
      return res.status(400).json({ code: 400, message: "Invalid user ID" });
    }
    const user = await adminModel.getUserDetailsById(userId);
    if (!user) {
      return res.status(404).json({ code: 404, message: "User not found" });
    }

    const months = await userModel.fetchUserMonths(userId);

    // return res.status(200).json({
    //   code: 200,
    //   message: "User details and progress fetched successfully",
    //   data: {
    //     user,
    //     months
    //   }
    // });

    res.render('admin/userDetails', { data: user, months });

  } catch (error) {
    console.error("Error in userDetails controller:", error);
    res.render('/');
    res.status(500).render('error', { message: 'Internal Server Error' });
  }
};

export const getUserMonthlyProgress = async (req, res) => {
  try {
    const { user_id, current_date } = req.query;

    const progress = await fetchProgress(user_id, current_date);

    return res.status(200).json({
      code: 200,
      message: "Monthly progress fetched successfully",
      data: progress
    });

  } catch (error) {
    console.error("Error in getUserMonthlyProgress:", error);
    return res.status(500).json({
      code: 500,
      message: "Internal server error"
    });
  }
};

export const addQuote = async (req, res) => {
  try {
    const { quote_text, author_name, date } = req.body;
    const result = await adminModel.insertQuote({ quote_text, author_name, date });
    if (result.exists) {
      return res.status(200).json({
        code: 200,
        message: "Quote already exists for this date.",
      });
    }
    return res.status(201).json({
      code: 200,
      message: "Quote added successfully.",
      quote_id: result.insertId,
    });
  } catch (error) {
    console.error("Error in addQuote controller:", error);
    return res.status(500).json({
      code: 500,
      message: "Internal Server Error.",
    });
  }
};

export const updateQuote = async (req, res) => {
  try {
    const { quote_id, quote_text, author_name, date } = req.body;
    const result = await adminModel.updateQuote({ quote_id, quote_text, author_name, date });
    if (result.exists) {
      return res.status(200).json({
        code: 200,
        message: "Another quote already exists for this date.",
      });
    }
    return res.status(200).json({
      code: 200,
      message: "Quote updated successfully.",
      data: { id: quote_id },
    });
  } catch (error) {
    console.error("Error in updateQuote controller:", error);
    return res.status(500).json({
      code: 500,
      message: "Internal Server Error",
    });
  }
};

export const deleteQuote = async (req, res) => {
  try {
    const { quote_id } = req.params;
    await adminModel.softDeleteQuote(quote_id);
    return res.status(200).json({
      code: 200,
      message: "Quote deleted successfully."
    });
  } catch (error) {
    console.log("Error in deleteQuote:", error);
    return res.status(500).json({
      code: 500,
      message: "Internal Server Error",
    });
  }
};

export const getAllQuotes = async (req, res) => {
  try {
    let { page = 1, limit = 10, search = "" } = req.query;
    page = Math.max(1, parseInt(page, 10));
    limit = Math.min(100, Math.max(1, parseInt(limit, 10)));
    const { data, total } = await adminModel.getQuotesList(page, limit, search.trim());
    return res.status(200).json({
      code: 200,
      message: "Quotes fetched successfully",
      data,
      pagination: {
        totalRecords: total,
        totalPages: Math.ceil(total / limit),
        currentPage: page,
        limit,
      },
    });
  } catch (error) {
    console.log("Error in getAllQuotes:", error);
    return res.status(500).json({
      code: 500,
      message: "Internal Server Error",
    });
  }
};

export const bulkUploadQuotes = async (req, res) => {
  try {
    if (!req.file)
      return res.status(400).json({ code: 400, message: "Excel file is required" });

    const workbook = XLSX.readFile(req.file.path);
    const sheetName = workbook.SheetNames[0];
    const worksheet = workbook.Sheets[sheetName];
    const rows = XLSX.utils.sheet_to_json(worksheet, { defval: "" });

    if (!rows.length) return res.status(400).json({ code: 400, message: "Excel file is empty" });

    const expectedHeaders = ["quote_text", "author_name", "date"];
    const actualHeaders = Object.keys(rows[0]).map(h => h.trim().toLowerCase());
    const missingHeaders = expectedHeaders.filter(h => !actualHeaders.includes(h));

    if (missingHeaders.length > 0)
      return res.status(400).json({
        code: 400,
        message: `Missing or invalid headers: ${missingHeaders.join(", ")}`,
      });

    const { insertedIds, invalidRows } = await adminModel.bulkUpsertQuotes(rows);

    return res.status(200).json({
      code: 200,
      message: "Bulk quotes processed successfully",
      totalRows: rows.length,
      insertedCount: insertedIds.length,
      invalidCount: invalidRows.length,
      insertedIds,
      invalidRows,
    });

  } catch (error) {
    console.error("❌ Error in bulkUploadQuotes controller:", error);
    return res.status(500).json({ code: 500, message: "Internal Server Error" });
  }
};